/*
 * International Telephone Input v15.0.0
 * https://github.com/jackocnr/intl-tel-input.git
 * Licensed under the MIT license
 */

// wrap in UMD
(function(factory) {
  var intlTelInput = factory(window, document);
  if (typeof module === "object" && module.exports) module.exports = intlTelInput; else window.intlTelInput = intlTelInput;
})(function(window, document, undefined) {
  "use strict";
  return function() {
      // Array of country objects for the flag dropdown.
      // Here is the criteria for the plugin to support a given country/territory
      // - It has an iso2 code: https://en.wikipedia.org/wiki/ISO_3166-1_alpha-2
      // - It has it's own country calling code (it is not a sub-region of another country): https://en.wikipedia.org/wiki/List_of_country_calling_codes
      // - It has a flag in the region-flags project: https://github.com/behdad/region-flags/tree/gh-pages/png
      // - It is supported by libphonenumber (it must be listed on this page): https://github.com/googlei18n/libphonenumber/blob/master/resources/ShortNumberMetadata.xml
      // Each country array has the following information:
      // [
      //    Country name,
      //    iso2 code,
      //    International dial code,
      //    Order (if >1 country with same dial code),
      //    Area codes
      // ]
      var allCountries = [ [ "Afghanistan (‫افغانستان‬‎)", "af", "93" ], [ "Albania (Shqipëri)", "al", "355" ], [ "Algeria (‫الجزائر‬‎)", "dz", "213" ], [ "American Samoa", "as", "1684" ], [ "Andorra", "ad", "376" ], [ "Angola", "ao", "244" ], [ "Anguilla", "ai", "1264" ], [ "Antigua and Barbuda", "ag", "1268" ], [ "Argentina", "ar", "54" ], [ "Armenia (Հայաստան)", "am", "374" ], [ "Aruba", "aw", "297" ], [ "Australia", "au", "61", 0 ], [ "Austria (Österreich)", "at", "43" ], [ "Azerbaijan (Azərbaycan)", "az", "994" ], [ "Bahamas", "bs", "1242" ], [ "Bahrain (‫البحرين‬‎)", "bh", "973" ], [ "Bangladesh (বাংলাদেশ)", "bd", "880" ], [ "Barbados", "bb", "1246" ], [ "Belarus (Беларусь)", "by", "375" ], [ "Belgium (België)", "be", "32" ], [ "Belize", "bz", "501" ], [ "Benin (Bénin)", "bj", "229" ], [ "Bermuda", "bm", "1441" ], [ "Bhutan (འབྲུག)", "bt", "975" ], [ "Bolivia", "bo", "591" ], [ "Bosnia and Herzegovina (Босна и Херцеговина)", "ba", "387" ], [ "Botswana", "bw", "267" ], [ "Brazil (Brasil)", "br", "55" ], [ "British Indian Ocean Territory", "io", "246" ], [ "British Virgin Islands", "vg", "1284" ], [ "Brunei", "bn", "673" ], [ "Bulgaria (България)", "bg", "359" ], [ "Burkina Faso", "bf", "226" ], [ "Burundi (Uburundi)", "bi", "257" ], [ "Cambodia (កម្ពុជា)", "kh", "855" ], [ "Cameroon (Cameroun)", "cm", "237" ], [ "Canada", "ca", "1", 1, [ "204", "226", "236", "249", "250", "289", "306", "343", "365", "387", "403", "416", "418", "431", "437", "438", "450", "506", "514", "519", "548", "579", "581", "587", "604", "613", "639", "647", "672", "705", "709", "742", "778", "780", "782", "807", "819", "825", "867", "873", "902", "905" ] ], [ "Cape Verde (Kabu Verdi)", "cv", "238" ], [ "Caribbean Netherlands", "bq", "599", 1 ], [ "Cayman Islands", "ky", "1345" ], [ "Central African Republic (République centrafricaine)", "cf", "236" ], [ "Chad (Tchad)", "td", "235" ], [ "Chile", "cl", "56" ], [ "China (中国)", "cn", "86" ], [ "Christmas Island", "cx", "61", 2 ], [ "Cocos (Keeling) Islands", "cc", "61", 1 ], [ "Colombia", "co", "57" ], [ "Comoros (‫جزر القمر‬‎)", "km", "269" ], [ "Congo (DRC) (Jamhuri ya Kidemokrasia ya Kongo)", "cd", "243" ], [ "Congo (Republic) (Congo-Brazzaville)", "cg", "242" ], [ "Cook Islands", "ck", "682" ], [ "Costa Rica", "cr", "506" ], [ "Côte d’Ivoire", "ci", "225" ], [ "Croatia (Hrvatska)", "hr", "385" ], [ "Cuba", "cu", "53" ], [ "Curaçao", "cw", "599", 0 ], [ "Cyprus (Κύπρος)", "cy", "357" ], [ "Czech Republic (Česká republika)", "cz", "420" ], [ "Denmark (Danmark)", "dk", "45" ], [ "Djibouti", "dj", "253" ], [ "Dominica", "dm", "1767" ], [ "Dominican Republic (República Dominicana)", "do", "1", 2, [ "809", "829", "849" ] ], [ "Ecuador", "ec", "593" ], [ "Egypt (‫مصر‬‎)", "eg", "20" ], [ "El Salvador", "sv", "503" ], [ "Equatorial Guinea (Guinea Ecuatorial)", "gq", "240" ], [ "Eritrea", "er", "291" ], [ "Estonia (Eesti)", "ee", "372" ], [ "Ethiopia", "et", "251" ], [ "Falkland Islands (Islas Malvinas)", "fk", "500" ], [ "Faroe Islands (Føroyar)", "fo", "298" ], [ "Fiji", "fj", "679" ], [ "Finland (Suomi)", "fi", "358", 0 ], [ "France", "fr", "33" ], [ "French Guiana (Guyane française)", "gf", "594" ], [ "French Polynesia (Polynésie française)", "pf", "689" ], [ "Gabon", "ga", "241" ], [ "Gambia", "gm", "220" ], [ "Georgia (საქართველო)", "ge", "995" ], [ "Germany (Deutschland)", "de", "49" ], [ "Ghana (Gaana)", "gh", "233" ], [ "Gibraltar", "gi", "350" ], [ "Greece (Ελλάδα)", "gr", "30" ], [ "Greenland (Kalaallit Nunaat)", "gl", "299" ], [ "Grenada", "gd", "1473" ], [ "Guadeloupe", "gp", "590", 0 ], [ "Guam", "gu", "1671" ], [ "Guatemala", "gt", "502" ], [ "Guernsey", "gg", "44", 1 ], [ "Guinea (Guinée)", "gn", "224" ], [ "Guinea-Bissau (Guiné Bissau)", "gw", "245" ], [ "Guyana", "gy", "592" ], [ "Haiti", "ht", "509" ], [ "Honduras", "hn", "504" ], [ "Hong Kong (香港)", "hk", "852" ], [ "Hungary (Magyarország)", "hu", "36" ], [ "Iceland (Ísland)", "is", "354" ], [ "India (भारत)", "in", "91" ], [ "Indonesia", "id", "62" ], [ "Iran (‫ایران‬‎)", "ir", "98" ], [ "Iraq (‫العراق‬‎)", "iq", "964" ], [ "Ireland", "ie", "353" ], [ "Isle of Man", "im", "44", 2 ], [ "Israel (‫ישראל‬‎)", "il", "972" ], [ "Italy (Italia)", "it", "39", 0 ], [ "Jamaica", "jm", "1", 4, [ "876", "658" ] ], [ "Japan (日本)", "jp", "81" ], [ "Jersey", "je", "44", 3 ], [ "Jordan (‫الأردن‬‎)", "jo", "962" ], [ "Kazakhstan (Казахстан)", "kz", "7", 1 ], [ "Kenya", "ke", "254" ], [ "Kiribati", "ki", "686" ], [ "Kosovo", "xk", "383" ], [ "Kuwait (‫الكويت‬‎)", "kw", "965" ], [ "Kyrgyzstan (Кыргызстан)", "kg", "996" ], [ "Laos (ລາວ)", "la", "856" ], [ "Latvia (Latvija)", "lv", "371" ], [ "Lebanon (‫لبنان‬‎)", "lb", "961" ], [ "Lesotho", "ls", "266" ], [ "Liberia", "lr", "231" ], [ "Libya (‫ليبيا‬‎)", "ly", "218" ], [ "Liechtenstein", "li", "423" ], [ "Lithuania (Lietuva)", "lt", "370" ], [ "Luxembourg", "lu", "352" ], [ "Macau (澳門)", "mo", "853" ], [ "Macedonia (FYROM) (Македонија)", "mk", "389" ], [ "Madagascar (Madagasikara)", "mg", "261" ], [ "Malawi", "mw", "265" ], [ "Malaysia", "my", "60" ], [ "Maldives", "mv", "960" ], [ "Mali", "ml", "223" ], [ "Malta", "mt", "356" ], [ "Marshall Islands", "mh", "692" ], [ "Martinique", "mq", "596" ], [ "Mauritania (‫موريتانيا‬‎)", "mr", "222" ], [ "Mauritius (Moris)", "mu", "230" ], [ "Mayotte", "yt", "262", 1 ], [ "Mexico (México)", "mx", "52" ], [ "Micronesia", "fm", "691" ], [ "Moldova (Republica Moldova)", "md", "373" ], [ "Monaco", "mc", "377" ], [ "Mongolia (Монгол)", "mn", "976" ], [ "Montenegro (Crna Gora)", "me", "382" ], [ "Montserrat", "ms", "1664" ], [ "Morocco (‫المغرب‬‎)", "ma", "212", 0 ], [ "Mozambique (Moçambique)", "mz", "258" ], [ "Myanmar (Burma) (မြန်မာ)", "mm", "95" ], [ "Namibia (Namibië)", "na", "264" ], [ "Nauru", "nr", "674" ], [ "Nepal (नेपाल)", "np", "977" ], [ "Netherlands (Nederland)", "nl", "31" ], [ "New Caledonia (Nouvelle-Calédonie)", "nc", "687" ], [ "New Zealand", "nz", "64" ], [ "Nicaragua", "ni", "505" ], [ "Niger (Nijar)", "ne", "227" ], [ "Nigeria", "ng", "234" ], [ "Niue", "nu", "683" ], [ "Norfolk Island", "nf", "672" ], [ "North Korea (조선 민주주의 인민 공화국)", "kp", "850" ], [ "Northern Mariana Islands", "mp", "1670" ], [ "Norway (Norge)", "no", "47", 0 ], [ "Oman (‫عُمان‬‎)", "om", "968" ], [ "Pakistan (‫پاکستان‬‎)", "pk", "92" ], [ "Palau", "pw", "680" ], [ "Palestine (‫فلسطين‬‎)", "ps", "970" ], [ "Panama (Panamá)", "pa", "507" ], [ "Papua New Guinea", "pg", "675" ], [ "Paraguay", "py", "595" ], [ "Peru (Perú)", "pe", "51" ], [ "Philippines", "ph", "63" ], [ "Poland (Polska)", "pl", "48" ], [ "Portugal", "pt", "351" ], [ "Puerto Rico", "pr", "1", 3, [ "787", "939" ] ], [ "Qatar (‫قطر‬‎)", "qa", "974" ], [ "Réunion (La Réunion)", "re", "262", 0 ], [ "Romania (România)", "ro", "40" ], [ "Russia (Россия)", "ru", "7", 0 ], [ "Rwanda", "rw", "250" ], [ "Saint Barthélemy", "bl", "590", 1 ], [ "Saint Helena", "sh", "290" ], [ "Saint Kitts and Nevis", "kn", "1869" ], [ "Saint Lucia", "lc", "1758" ], [ "Saint Martin (Saint-Martin (partie française))", "mf", "590", 2 ], [ "Saint Pierre and Miquelon (Saint-Pierre-et-Miquelon)", "pm", "508" ], [ "Saint Vincent and the Grenadines", "vc", "1784" ], [ "Samoa", "ws", "685" ], [ "San Marino", "sm", "378" ], [ "São Tomé and Príncipe (São Tomé e Príncipe)", "st", "239" ], [ "Saudi Arabia (‫المملكة العربية السعودية‬‎)", "sa", "966" ], [ "Senegal (Sénégal)", "sn", "221" ], [ "Serbia (Србија)", "rs", "381" ], [ "Seychelles", "sc", "248" ], [ "Sierra Leone", "sl", "232" ], [ "Singapore", "sg", "65" ], [ "Sint Maarten", "sx", "1721" ], [ "Slovakia (Slovensko)", "sk", "421" ], [ "Slovenia (Slovenija)", "si", "386" ], [ "Solomon Islands", "sb", "677" ], [ "Somalia (Soomaaliya)", "so", "252" ], [ "South Africa", "za", "27" ], [ "South Korea (대한민국)", "kr", "82" ], [ "South Sudan (‫جنوب السودان‬‎)", "ss", "211" ], [ "Spain (España)", "es", "34" ], [ "Sri Lanka (ශ්‍රී ලංකාව)", "lk", "94" ], [ "Sudan (‫السودان‬‎)", "sd", "249" ], [ "Suriname", "sr", "597" ], [ "Svalbard and Jan Mayen", "sj", "47", 1 ], [ "Swaziland", "sz", "268" ], [ "Sweden (Sverige)", "se", "46" ], [ "Switzerland (Schweiz)", "ch", "41" ], [ "Syria (‫سوريا‬‎)", "sy", "963" ], [ "Taiwan (台灣)", "tw", "886" ], [ "Tajikistan", "tj", "992" ], [ "Tanzania", "tz", "255" ], [ "Thailand (ไทย)", "th", "66" ], [ "Timor-Leste", "tl", "670" ], [ "Togo", "tg", "228" ], [ "Tokelau", "tk", "690" ], [ "Tonga", "to", "676" ], [ "Trinidad and Tobago", "tt", "1868" ], [ "Tunisia (‫تونس‬‎)", "tn", "216" ], [ "Turkey (Türkiye)", "tr", "90" ], [ "Turkmenistan", "tm", "993" ], [ "Turks and Caicos Islands", "tc", "1649" ], [ "Tuvalu", "tv", "688" ], [ "U.S. Virgin Islands", "vi", "1340" ], [ "Uganda", "ug", "256" ], [ "Ukraine (Україна)", "ua", "380" ], [ "United Arab Emirates (‫الإمارات العربية المتحدة‬‎)", "ae", "971" ], [ "United Kingdom", "gb", "44", 0 ], [ "United States", "us", "1", 0 ], [ "Uruguay", "uy", "598" ], [ "Uzbekistan (Oʻzbekiston)", "uz", "998" ], [ "Vanuatu", "vu", "678" ], [ "Vatican City (Città del Vaticano)", "va", "39", 1 ], [ "Venezuela", "ve", "58" ], [ "Vietnam (Việt Nam)", "vn", "84" ], [ "Wallis and Futuna (Wallis-et-Futuna)", "wf", "681" ], [ "Western Sahara (‫الصحراء الغربية‬‎)", "eh", "212", 1 ], [ "Yemen (‫اليمن‬‎)", "ye", "967" ], [ "Zambia", "zm", "260" ], [ "Zimbabwe", "zw", "263" ], [ "Åland Islands", "ax", "358", 1 ] ];
      // loop over all of the countries above, restructuring the data to be objects with named keys
      for (var i = 0; i < allCountries.length; i++) {
          var c = allCountries[i];
          allCountries[i] = {
              name: c[0],
              iso2: c[1],
              dialCode: c[2],
              priority: c[3] || 0,
              areaCodes: c[4] || null
          };
      }
      "use strict";
      function _classCallCheck(instance, Constructor) {
          if (!(instance instanceof Constructor)) {
              throw new TypeError("Cannot call a class as a function");
          }
      }
      function _defineProperties(target, props) {
          for (var i = 0; i < props.length; i++) {
              var descriptor = props[i];
              descriptor.enumerable = descriptor.enumerable || false;
              descriptor.configurable = true;
              if ("value" in descriptor) descriptor.writable = true;
              Object.defineProperty(target, descriptor.key, descriptor);
          }
      }
      function _createClass(Constructor, protoProps, staticProps) {
          if (protoProps) _defineProperties(Constructor.prototype, protoProps);
          if (staticProps) _defineProperties(Constructor, staticProps);
          return Constructor;
      }
      window.intlTelInputGlobals = {
          getInstance: function getInstance(input) {
              var id = input.getAttribute("data-intl-tel-input-id");
              return window.intlTelInputGlobals.instances[id];
          },
          instances: {}
      };
      // these vars persist through all instances of the plugin
      var id = 0;
      var defaults = {
          // whether or not to allow the dropdown
          allowDropdown: true,
          // if there is just a dial code in the input: remove it on blur
          autoHideDialCode: true,
          // add a placeholder in the input with an example number for the selected country
          autoPlaceholder: "polite",
          // modify the parentClass
          customContainer: "",
          // modify the auto placeholder
          customPlaceholder: null,
          // append menu to specified element
          dropdownContainer: null,
          // don't display these countries
          excludeCountries: [],
          // format the input value during initialisation and on setNumber
          formatOnDisplay: true,
          // geoIp lookup function
          geoIpLookup: null,
          // inject a hidden input with this name, and on submit, populate it with the result of getNumber
          hiddenInput: "",
          // initial country
          initialCountry: "",
          // localized country names e.g. { 'de': 'Deutschland' }
          localizedCountries: null,
          // don't insert international dial codes
          nationalMode: true,
          // display only these countries
          onlyCountries: [],
          // number type to use for placeholders
          placeholderNumberType: "MOBILE",
          // the countries at the top of the list. defaults to united states and united kingdom
          preferredCountries: [ "us", "gb" ],
          // display the country dial code next to the selected flag so it's not part of the typed number
          separateDialCode: false,
          // specify the path to the libphonenumber script to enable validation/formatting
          utilsScript: ""
      };
      // https://en.wikipedia.org/wiki/List_of_North_American_Numbering_Plan_area_codes#Non-geographic_area_codes
      var regionlessNanpNumbers = [ "800", "822", "833", "844", "855", "866", "877", "880", "881", "882", "883", "884", "885", "886", "887", "888", "889" ];
      // keep track of if the window.load event has fired as impossible to check after the fact
      window.addEventListener("load", function() {
          // UPDATE: use a public static field so we can fudge it in the tests
          window.intlTelInputGlobals.windowLoaded = true;
      });
      // utility function to iterate over an object. can't use Object.entries or native forEach because
      // of IE11
      var forEachProp = function forEachProp(obj, callback) {
          var keys = Object.keys(obj);
          for (var i = 0; i < keys.length; i++) {
              callback(keys[i], obj[keys[i]]);
          }
      };
      // run a method on each instance of the plugin
      var forEachInstance = function forEachInstance(method) {
          forEachProp(window.intlTelInputGlobals.instances, function(key) {
              window.intlTelInputGlobals.instances[key][method]();
          });
      };
      // this is our plugin class that we will create an instance of
      // eslint-disable-next-line no-unused-vars
      var Iti = /*#__PURE__*/
      function() {
          function Iti(input, options) {
              var _this = this;
              _classCallCheck(this, Iti);
              this.id = id++;
              this.telInput = input;
              this.activeItem = null;
              this.highlightedItem = null;
              // process specified options / defaults
              // alternative to Object.assign, which isn't supported by IE11
              var customOptions = options || {};
              this.options = {};
              forEachProp(defaults, function(key, value) {
                  _this.options[key] = customOptions.hasOwnProperty(key) ? customOptions[key] : value;
              });
              this.hadInitialPlaceholder = Boolean(input.getAttribute("placeholder"));
          }
          _createClass(Iti, [ {
              key: "_init",
              value: function _init() {
                  var _this2 = this;
                  // if in nationalMode, disable options relating to dial codes
                  if (this.options.nationalMode) this.options.autoHideDialCode = false;
                  // if separateDialCode then doesn't make sense to A) insert dial code into input
                  // (autoHideDialCode), and B) display national numbers (because we're displaying the country
                  // dial code next to them)
                  if (this.options.separateDialCode) {
                      this.options.autoHideDialCode = this.options.nationalMode = false;
                  }
                  // we cannot just test screen size as some smartphones/website meta tags will report desktop
                  // resolutions
                  // Note: for some reason jasmine breaks if you put this in the main Plugin function with the
                  // rest of these declarations
                  // Note: to target Android Mobiles (and not Tablets), we must find 'Android' and 'Mobile'
                  this.isMobile = /Android.+Mobile|webOS|iPhone|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
                  if (this.isMobile) {
                      // trigger the mobile dropdown css
                      document.body.classList.add("iti-mobile");
                      // on mobile, we want a full screen dropdown, so we must append it to the body
                      if (!this.options.dropdownContainer) this.options.dropdownContainer = document.body;
                  }
                  // these promises get resolved when their individual requests complete
                  // this way the dev can do something like iti.promise.then(...) to know when all requests are
                  // complete
                  if (typeof Promise !== "undefined") {
                      var autoCountryPromise = new Promise(function(resolve, reject) {
                          _this2.resolveAutoCountryPromise = resolve;
                          _this2.rejectAutoCountryPromise = reject;
                      });
                      var utilsScriptPromise = new Promise(function(resolve, reject) {
                          _this2.resolveUtilsScriptPromise = resolve;
                          _this2.rejectUtilsScriptPromise = reject;
                      });
                      this.promise = Promise.all([ autoCountryPromise, utilsScriptPromise ]);
                  } else {
                      // prevent errors when Promise doesn't exist
                      this.resolveAutoCountryPromise = this.rejectAutoCountryPromise = function() {};
                      this.resolveUtilsScriptPromise = this.rejectUtilsScriptPromise = function() {};
                  }
                  // in various situations there could be no country selected initially, but we need to be able
                  // to assume this variable exists
                  this.selectedCountryData = {};
                  // process all the data: onlyCountries, excludeCountries, preferredCountries etc
                  this._processCountryData();
                  // generate the markup
                  this._generateMarkup();
                  // set the initial state of the input value and the selected flag
                  this._setInitialState();
                  // start all of the event listeners: autoHideDialCode, input keydown, selectedFlag click
                  this._initListeners();
                  // utils script, and auto country
                  this._initRequests();
              }
          }, {
              key: "_processCountryData",
              value: function _processCountryData() {
                  // process onlyCountries or excludeCountries array if present
                  this._processAllCountries();
                  // process the countryCodes map
                  this._processCountryCodes();
                  // process the preferredCountries
                  this._processPreferredCountries();
                  // translate countries according to localizedCountries option
                  if (this.options.localizedCountries) this._translateCountriesByLocale();
                  // sort countries by name
                  if (this.options.onlyCountries.length || this.options.localizedCountries) {
                      this.countries.sort(this._countryNameSort);
                  }
              }
          }, {
              key: "_addCountryCode",
              value: function _addCountryCode(iso2, dialCode, priority) {
                  if (!this.countryCodes.hasOwnProperty(dialCode)) {
                      this.countryCodes[dialCode] = [];
                  }
                  var index = priority || 0;
                  this.countryCodes[dialCode][index] = iso2;
              }
          }, {
              key: "_processAllCountries",
              value: function _processAllCountries() {
                  if (this.options.onlyCountries.length) {
                      var lowerCaseOnlyCountries = this.options.onlyCountries.map(function(country) {
                          return country.toLowerCase();
                      });
                      this.countries = allCountries.filter(function(country) {
                          return lowerCaseOnlyCountries.indexOf(country.iso2) > -1;
                      });
                  } else if (this.options.excludeCountries.length) {
                      var lowerCaseExcludeCountries = this.options.excludeCountries.map(function(country) {
                          return country.toLowerCase();
                      });
                      this.countries = allCountries.filter(function(country) {
                          return lowerCaseExcludeCountries.indexOf(country.iso2) === -1;
                      });
                  } else {
                      this.countries = allCountries;
                  }
              }
          }, {
              key: "_translateCountriesByLocale",
              value: function _translateCountriesByLocale() {
                  for (var i = 0; i < this.countries.length; i++) {
                      var iso = this.countries[i].iso2.toLowerCase();
                      if (this.options.localizedCountries.hasOwnProperty(iso)) {
                          this.countries[i].name = this.options.localizedCountries[iso];
                      }
                  }
              }
          }, {
              key: "_countryNameSort",
              value: function _countryNameSort(a, b) {
                  return a.name.localeCompare(b.name);
              }
          }, {
              key: "_processCountryCodes",
              value: function _processCountryCodes() {
                  this.countryCodes = {};
                  for (var i = 0; i < this.countries.length; i++) {
                      var c = this.countries[i];
                      this._addCountryCode(c.iso2, c.dialCode, c.priority);
                      // area codes
                      if (c.areaCodes) {
                          for (var j = 0; j < c.areaCodes.length; j++) {
                              // full dial code is country code + dial code
                              this._addCountryCode(c.iso2, c.dialCode + c.areaCodes[j]);
                          }
                      }
                  }
              }
          }, {
              key: "_processPreferredCountries",
              value: function _processPreferredCountries() {
                  this.preferredCountries = [];
                  for (var i = 0; i < this.options.preferredCountries.length; i++) {
                      var countryCode = this.options.preferredCountries[i].toLowerCase();
                      var countryData = this._getCountryData(countryCode, false, true);
                      if (countryData) this.preferredCountries.push(countryData);
                  }
              }
          }, {
              key: "_createEl",
              value: function _createEl(name, attrs, container) {
                  var el = document.createElement(name);
                  if (attrs) forEachProp(attrs, function(key, value) {
                      return el.setAttribute(key, value);
                  });
                  if (container) container.appendChild(el);
                  return el;
              }
          }, {
              key: "_generateMarkup",
              value: function _generateMarkup() {
                  // prevent autocomplete as there's no safe, cross-browser event we can react to, so it can
                  // easily put the plugin in an inconsistent state e.g. the wrong flag selected for the
                  // autocompleted number, which on submit could mean wrong number is saved (esp in nationalMode)
                  this.telInput.setAttribute("autocomplete", "off");
                  // containers (mostly for positioning)
                  var parentClass = "intl-tel-input";
                  if (this.options.allowDropdown) parentClass += " allow-dropdown";
                  if (this.options.separateDialCode) parentClass += " separate-dial-code";
                  if (this.options.customContainer) {
                      parentClass += " ";
                      parentClass += this.options.customContainer;
                  }
                  var wrapper = this._createEl("div", {
                      "class": parentClass
                  });
                  this.telInput.parentNode.insertBefore(wrapper, this.telInput);
                  this.flagsContainer = this._createEl("div", {
                      "class": "flag-container"
                  }, wrapper);
                  wrapper.appendChild(this.telInput);
                  // selected flag (displayed to left of input)
                  this.selectedFlag = this._createEl("div", {
                      "class": "selected-flag",
                      role: "combobox",
                      "aria-owns": "country-listbox"
                  }, this.flagsContainer);
                  this.selectedFlagInner = this._createEl("div", {
                      "class": "selected-flag-line"
                  }, this.selectedFlag);
                  this.selectedFlagInner = this._createEl("div", {
                      "class": "iti-flag"
                  }, this.selectedFlag);
                  if (this.options.separateDialCode) {
                      this.selectedDialCode = this._createEl("div", {
                          "class": "selected-dial-code"
                      }, this.selectedFlag);
                  }
                  if (this.options.allowDropdown) {
                      // make element focusable and tab navigable
                      this.selectedFlag.setAttribute("tabindex", "0");
                      this.dropdownArrow = this._createEl("div", {
                          "class": "iti-arrow"
                      }, this.selectedFlag);
                      // country dropdown: preferred countries, then divider, then all countries
                      this.countryList = this._createEl("ul", {
                          "class": "country-list hide",
                          id: "country-listbox",
                          "aria-expanded": "false",
                          role: "listbox"
                      });
                      if (this.preferredCountries.length) {
                          this._appendListItems(this.preferredCountries, "preferred");
                          this._createEl("li", {
                              "class": "divider",
                              role: "separator",
                              "aria-disabled": "true"
                          }, this.countryList);
                      }
                      this._appendListItems(this.countries, "standard");
                      // create dropdownContainer markup
                      if (this.options.dropdownContainer) {
                          this.dropdown = this._createEl("div", {
                              "class": "intl-tel-input iti-container"
                          });
                          this.dropdown.appendChild(this.countryList);
                      } else {
                          this.flagsContainer.appendChild(this.countryList);
                      }
                  }
                  if (this.options.hiddenInput) {
                      var hiddenInputName = this.options.hiddenInput;
                      var name = this.telInput.getAttribute("name");
                      if (name) {
                          var i = name.lastIndexOf("[");
                          // if input name contains square brackets, then give the hidden input the same name,
                          // replacing the contents of the last set of brackets with the given hiddenInput name
                          if (i !== -1) hiddenInputName = "".concat(name.substr(0, i), "[").concat(hiddenInputName, "]");
                      }
                      this.hiddenInput = this._createEl("input", {
                          type: "hidden",
                          name: hiddenInputName
                      });
                      wrapper.appendChild(this.hiddenInput);
                  }
              }
          }, {
              key: "_appendListItems",
              value: function _appendListItems(countries, className) {
                  // we create so many DOM elements, it is faster to build a temp string
                  // and then add everything to the DOM in one go at the end
                  var tmp = "";
                  // for each country
                  for (var i = 0; i < countries.length; i++) {
                      var c = countries[i];
                      // open the list item
                      tmp += "<li class='country ".concat(className, "' id='iti-item-").concat(c.iso2, "' role='option' data-dial-code='").concat(c.dialCode, "' data-country-code='").concat(c.iso2, "'>");
                      // add the flag
                      tmp += "<div class='flag-box'><div class='iti-flag ".concat(c.iso2, "'></div></div>");
                      // and the country name and dial code
                      tmp += "<span class='country-name'>".concat(c.name, "</span>");
                      tmp += "<span class='dial-code'>+".concat(c.dialCode, "</span>");
                      // close the list item
                      tmp += "</li>";
                  }
                  this.countryList.insertAdjacentHTML("beforeend", tmp);
              }
          }, {
              key: "_setInitialState",
              value: function _setInitialState() {
                  var val = this.telInput.value;
                  var dialCode = this._getDialCode(val);
                  var isRegionlessNanp = this._isRegionlessNanp(val);
                  var _this$options = this.options, initialCountry = _this$options.initialCountry, nationalMode = _this$options.nationalMode, autoHideDialCode = _this$options.autoHideDialCode, separateDialCode = _this$options.separateDialCode;
                  // if we already have a dial code, and it's not a regionlessNanp, we can go ahead and set the
                  // flag, else fall back to the default country
                  if (dialCode && !isRegionlessNanp) {
                      this._updateFlagFromNumber(val);
                  } else if (initialCountry !== "auto") {
                      // see if we should select a flag
                      if (initialCountry) {
                          this._setFlag(initialCountry.toLowerCase());
                      } else {
                          if (dialCode && isRegionlessNanp) {
                              // has intl dial code, is regionless nanp, and no initialCountry, so default to US
                              this._setFlag("us");
                          } else {
                              // no dial code and no initialCountry, so default to first in list
                              this.defaultCountry = this.preferredCountries.length ? this.preferredCountries[0].iso2 : this.countries[0].iso2;
                              if (!val) {
                                  this._setFlag(this.defaultCountry);
                              }
                          }
                      }
                      // if empty and no nationalMode and no autoHideDialCode then insert the default dial code
                      if (!val && !nationalMode && !autoHideDialCode && !separateDialCode) {
                          this.telInput.value = "+".concat(this.selectedCountryData.dialCode);
                      }
                  }
                  // NOTE: if initialCountry is set to auto, that will be handled separately
                  // format - note this wont be run after _updateDialCode as that's only called if no val
                  if (val) this._updateValFromNumber(val);
              }
          }, {
              key: "_initListeners",
              value: function _initListeners() {
                  this._initKeyListeners();
                  if (this.options.autoHideDialCode) this._initBlurListeners();
                  if (this.options.allowDropdown) this._initDropdownListeners();
                  if (this.hiddenInput) this._initHiddenInputListener();
              }
          }, {
              key: "_initHiddenInputListener",
              value: function _initHiddenInputListener() {
                  var _this3 = this;
                  this._handleHiddenInputSubmit = function() {
                      _this3.hiddenInput.value = _this3.getNumber();
                  };
                  if (this.telInput.form) this.telInput.form.addEventListener("submit", this._handleHiddenInputSubmit);
              }
          }, {
              key: "_getClosestLabel",
              value: function _getClosestLabel() {
                  var el = this.telInput;
                  while (el && el.tagName !== "LABEL") {
                      el = el.parentNode;
                  }
                  return el;
              }
          }, {
              key: "_initDropdownListeners",
              value: function _initDropdownListeners() {
                  var _this4 = this;
                  // hack for input nested inside label (which is valid markup): clicking the selected-flag to
                  // open the dropdown would then automatically trigger a 2nd click on the input which would
                  // close it again
                  this._handleLabelClick = function(e) {
                      // if the dropdown is closed, then focus the input, else ignore the click
                      if (_this4.countryList.classList.contains("hide")) _this4.telInput.focus(); else e.preventDefault();
                  };
                  var label = this._getClosestLabel();
                  if (label) label.addEventListener("click", this._handleLabelClick);
                  // toggle country dropdown on click
                  this._handleClickSelectedFlag = function() {
                      // only intercept this event if we're opening the dropdown
                      // else let it bubble up to the top ("click-off-to-close" listener)
                      // we cannot just stopPropagation as it may be needed to close another instance
                      if (_this4.countryList.classList.contains("hide") && !_this4.telInput.disabled && !_this4.telInput.readOnly) {
                          _this4._showDropdown();
                      }
                  };
                  this.selectedFlag.addEventListener("click", this._handleClickSelectedFlag);
                  // open dropdown list if currently focused
                  this._handleFlagsContainerKeydown = function(e) {
                      var isDropdownHidden = _this4.countryList.classList.contains("hide");
                      if (isDropdownHidden && [ "ArrowUp", "ArrowDown", " ", "Enter" ].indexOf(e.key) !== -1) {
                          // prevent form from being submitted if "ENTER" was pressed
                          e.preventDefault();
                          // prevent event from being handled again by document
                          e.stopPropagation();
                          _this4._showDropdown();
                      }
                      // allow navigation from dropdown to input on TAB
                      if (e.key === "Tab") _this4._closeDropdown();
                  };
                  this.flagsContainer.addEventListener("keydown", this._handleFlagsContainerKeydown);
              }
          }, {
              key: "_initRequests",
              value: function _initRequests() {
                  var _this5 = this;
                  // if the user has specified the path to the utils script, fetch it on window.load, else resolve
                  if (this.options.utilsScript && !window.intlTelInputUtils) {
                      // if the plugin is being initialised after the window.load event has already been fired
                      if (window.intlTelInputGlobals.windowLoaded) {
                          window.intlTelInputGlobals.loadUtils(this.options.utilsScript);
                      } else {
                          // wait until the load event so we don't block any other requests e.g. the flags image
                          window.addEventListener("load", function() {
                              window.intlTelInputGlobals.loadUtils(_this5.options.utilsScript);
                          });
                      }
                  } else this.resolveUtilsScriptPromise();
                  if (this.options.initialCountry === "auto") this._loadAutoCountry(); else this.resolveAutoCountryPromise();
              }
          }, {
              key: "_loadAutoCountry",
              value: function _loadAutoCountry() {
                  // 3 options:
                  // 1) already loaded (we're done)
                  // 2) not already started loading (start)
                  // 3) already started loading (do nothing - just wait for loading callback to fire)
                  if (window.intlTelInputGlobals.autoCountry) {
                      this.handleAutoCountry();
                  } else if (!window.intlTelInputGlobals.startedLoadingAutoCountry) {
                      // don't do this twice!
                      window.intlTelInputGlobals.startedLoadingAutoCountry = true;
                      if (typeof this.options.geoIpLookup === "function") {
                          this.options.geoIpLookup(function(countryCode) {
                              window.intlTelInputGlobals.autoCountry = countryCode.toLowerCase();
                              // tell all instances the auto country is ready
                              // TODO: this should just be the current instances
                              // UPDATE: use setTimeout in case their geoIpLookup function calls this callback straight
                              // away (e.g. if they have already done the geo ip lookup somewhere else). Using
                              // setTimeout means that the current thread of execution will finish before executing
                              // this, which allows the plugin to finish initialising.
                              setTimeout(function() {
                                  return forEachInstance("handleAutoCountry");
                              });
                          }, function() {
                              return forEachInstance("rejectAutoCountryPromise");
                          });
                      }
                  }
              }
          }, {
              key: "_initKeyListeners",
              value: function _initKeyListeners() {
                  var _this6 = this;
                  // update flag on keyup
                  this._handleKeyupEvent = function() {
                      if (_this6._updateFlagFromNumber(_this6.telInput.value)) {
                          _this6._triggerCountryChange();
                      }
                  };
                  this.telInput.addEventListener("keyup", this._handleKeyupEvent);
                  // update flag on cut/paste events (now supported in all major browsers)
                  this._handleClipboardEvent = function() {
                      // hack because "paste" event is fired before input is updated
                      setTimeout(_this6._handleKeyupEvent);
                  };
                  this.telInput.addEventListener("cut", this._handleClipboardEvent);
                  this.telInput.addEventListener("paste", this._handleClipboardEvent);
              }
          }, {
              key: "_cap",
              value: function _cap(number) {
                  var max = this.telInput.getAttribute("maxlength");
                  return max && number.length > max ? number.substr(0, max) : number;
              }
          }, {
              key: "_initBlurListeners",
              value: function _initBlurListeners() {
                  var _this7 = this;
                  // on blur or form submit: if just a dial code then remove it
                  this._handleSubmitOrBlurEvent = function() {
                      _this7._removeEmptyDialCode();
                  };
                  if (this.telInput.form) this.telInput.form.addEventListener("submit", this._handleSubmitOrBlurEvent);
                  this.telInput.addEventListener("blur", this._handleSubmitOrBlurEvent);
              }
          }, {
              key: "_removeEmptyDialCode",
              value: function _removeEmptyDialCode() {
                  if (this.telInput.value.charAt(0) === "+") {
                      var numeric = this._getNumeric(this.telInput.value);
                      // if just a plus, or if just a dial code
                      if (!numeric || this.selectedCountryData.dialCode === numeric) {
                          this.telInput.value = "";
                      }
                  }
              }
          }, {
              key: "_getNumeric",
              value: function _getNumeric(s) {
                  return s.replace(/\D/g, "");
              }
          }, {
              key: "_trigger",
              value: function _trigger(name) {
                  // have to use old school document.createEvent as IE11 doesn't support `new Event()` syntax
                  var e = document.createEvent("Event");
                  e.initEvent(name, true, true);
                  // can bubble, and is cancellable
                  this.telInput.dispatchEvent(e);
              }
          }, {
              key: "_showDropdown",
              value: function _showDropdown() {
                  this.countryList.classList.remove("hide");
                  this.countryList.setAttribute("aria-expanded", "true");
                  this._setDropdownPosition();
                  // update highlighting and scroll to active list item
                  if (this.activeItem) {
                      this._highlightListItem(this.activeItem);
                      this._scrollTo(this.activeItem);
                  }
                  // bind all the dropdown-related listeners: mouseover, click, click-off, keydown
                  this._bindDropdownListeners();
                  // update the arrow
                  this.dropdownArrow.classList.add("up");
                  this._trigger("open:countrydropdown");
              }
          }, {
              key: "_toggleClass",
              value: function _toggleClass(el, className, shouldHaveClass) {
                  if (shouldHaveClass && !el.classList.contains(className)) el.classList.add(className); else if (!shouldHaveClass && el.classList.contains(className)) el.classList.remove(className);
              }
          }, {
              key: "_setDropdownPosition",
              value: function _setDropdownPosition() {
                  var _this8 = this;
                  if (this.options.dropdownContainer) {
                      this.options.dropdownContainer.appendChild(this.dropdown);
                  }
                  if (!this.isMobile) {
                      var pos = this.telInput.getBoundingClientRect();
                      // windowTop from https://stackoverflow.com/a/14384091/217866
                      var windowTop = window.pageYOffset || document.documentElement.scrollTop;
                      var inputTop = pos.top + windowTop;
                      var dropdownHeight = this.countryList.offsetHeight;
                      // dropdownFitsBelow = (dropdownBottom < windowBottom)
                      var dropdownFitsBelow = inputTop + this.telInput.offsetHeight + dropdownHeight < windowTop + window.innerHeight;
                      var dropdownFitsAbove = inputTop - dropdownHeight > windowTop;
                      // by default, the dropdown will be below the input. If we want to position it above the
                      // input, we add the dropup class.
                      this._toggleClass(this.countryList, "dropup", !dropdownFitsBelow && dropdownFitsAbove);
                      // if dropdownContainer is enabled, calculate postion
                      if (this.options.dropdownContainer) {
                          // by default the dropdown will be directly over the input because it's not in the flow.
                          // If we want to position it below, we need to add some extra top value.
                          var extraTop = !dropdownFitsBelow && dropdownFitsAbove ? 0 : this.telInput.offsetHeight;
                          // calculate placement
                          this.dropdown.style.top = "".concat(inputTop + extraTop, "px");
                          this.dropdown.style.left = "".concat(pos.left + document.body.scrollLeft, "px");
                          // close menu on window scroll
                          this._handleWindowScroll = function() {
                              return _this8._closeDropdown();
                          };
                          window.addEventListener("scroll", this._handleWindowScroll);
                      }
                  }
              }
          }, {
              key: "_getClosestListItem",
              value: function _getClosestListItem(target) {
                  var el = target;
                  while (el && el !== this.countryList && !el.classList.contains("country")) {
                      el = el.parentNode;
                  }
                  // if we reached the countryList element, then return null
                  return el === this.countryList ? null : el;
              }
          }, {
              key: "_bindDropdownListeners",
              value: function _bindDropdownListeners() {
                  var _this9 = this;
                  // when mouse over a list item, just highlight that one
                  // we add the class "highlight", so if they hit "enter" we know which one to select
                  this._handleMouseoverCountryList = function(e) {
                      // handle event delegation, as we're listening for this event on the countryList
                      var listItem = _this9._getClosestListItem(e.target);
                      if (listItem) _this9._highlightListItem(listItem);
                  };
                  this.countryList.addEventListener("mouseover", this._handleMouseoverCountryList);
                  // listen for country selection
                  this._handleClickCountryList = function(e) {
                      var listItem = _this9._getClosestListItem(e.target);
                      if (listItem) _this9._selectListItem(listItem);
                  };
                  this.countryList.addEventListener("click", this._handleClickCountryList);
                  // click off to close
                  // (except when this initial opening click is bubbling up)
                  // we cannot just stopPropagation as it may be needed to close another instance
                  var isOpening = true;
                  this._handleClickOffToClose = function() {
                      if (!isOpening) _this9._closeDropdown();
                      isOpening = false;
                  };
                  document.documentElement.addEventListener("click", this._handleClickOffToClose);
                  // listen for up/down scrolling, enter to select, or letters to jump to country name.
                  // use keydown as keypress doesn't fire for non-char keys and we want to catch if they
                  // just hit down and hold it to scroll down (no keyup event).
                  // listen on the document because that's where key events are triggered if no input has focus
                  var query = "";
                  var queryTimer = null;
                  this._handleKeydownOnDropdown = function(e) {
                      // prevent down key from scrolling the whole page,
                      // and enter key from submitting a form etc
                      e.preventDefault();
                      // up and down to navigate
                      if (e.key === "ArrowUp" || e.key === "ArrowDown") _this9._handleUpDownKey(e.key); else if (e.key === "Enter") _this9._handleEnterKey(); else if (e.key === "Escape") _this9._closeDropdown(); else if (/^[a-zA-ZÀ-ÿ ]$/.test(e.key)) {
                          // jump to countries that start with the query string
                          if (queryTimer) clearTimeout(queryTimer);
                          query += e.key.toLowerCase();
                          _this9._searchForCountry(query);
                          // if the timer hits 1 second, reset the query
                          queryTimer = setTimeout(function() {
                              query = "";
                          }, 1e3);
                      }
                  };
                  document.addEventListener("keydown", this._handleKeydownOnDropdown);
              }
          }, {
              key: "_handleUpDownKey",
              value: function _handleUpDownKey(key) {
                  var next = key === "ArrowUp" ? this.highlightedItem.previousElementSibling : this.highlightedItem.nextElementSibling;
                  if (next) {
                      // skip the divider
                      if (next.classList.contains("divider")) {
                          next = key === "ArrowUp" ? next.previousElementSibling : next.nextElementSibling;
                      }
                      this._highlightListItem(next);
                      this._scrollTo(next);
                  }
              }
          }, {
              key: "_handleEnterKey",
              value: function _handleEnterKey() {
                  if (this.highlightedItem) this._selectListItem(this.highlightedItem);
              }
          }, {
              key: "_searchForCountry",
              value: function _searchForCountry(query) {
                  for (var i = 0; i < this.countries.length; i++) {
                      if (this._startsWith(this.countries[i].name, query)) {
                          var listItem = this.countryList.querySelector("#iti-item-".concat(this.countries[i].iso2));
                          // update highlighting and scroll
                          this._highlightListItem(listItem);
                          this._scrollTo(listItem, true);
                          break;
                      }
                  }
              }
          }, {
              key: "_startsWith",
              value: function _startsWith(a, b) {
                  return a.substr(0, b.length).toLowerCase() === b;
              }
          }, {
              key: "_updateValFromNumber",
              value: function _updateValFromNumber(originalNumber) {
                  var number = originalNumber;
                  if (this.options.formatOnDisplay && window.intlTelInputUtils && this.selectedCountryData) {
                      var useNational = !this.options.separateDialCode && (this.options.nationalMode || number.charAt(0) !== "+");
                      var _intlTelInputUtils$nu = intlTelInputUtils.numberFormat, NATIONAL = _intlTelInputUtils$nu.NATIONAL, INTERNATIONAL = _intlTelInputUtils$nu.INTERNATIONAL;
                      var format = useNational ? NATIONAL : INTERNATIONAL;
                      number = intlTelInputUtils.formatNumber(number, this.selectedCountryData.iso2, format);
                  }
                  number = this._beforeSetNumber(number);
                  this.telInput.value = number;
              }
          }, {
              key: "_updateFlagFromNumber",
              value: function _updateFlagFromNumber(originalNumber) {
                  // if we're in nationalMode and we already have US/Canada selected, make sure the number starts
                  // with a +1 so _getDialCode will be able to extract the area code
                  // update: if we dont yet have selectedCountryData, but we're here (trying to update the flag
                  // from the number), that means we're initialising the plugin with a number that already has a
                  // dial code, so fine to ignore this bit
                  var number = originalNumber;
                  var isNanp = this.selectedCountryData.dialCode === "1";
                  if (number && this.options.nationalMode && isNanp && number.charAt(0) !== "+") {
                      if (number.charAt(0) !== "1") number = "1".concat(number);
                      number = "+".concat(number);
                  }
                  // try and extract valid dial code from input
                  var dialCode = this._getDialCode(number);
                  var numeric = this._getNumeric(number);
                  var countryCode = null;
                  if (dialCode) {
                      // check if one of the matching countries is already selected
                      var countryCodes = this.countryCodes[this._getNumeric(dialCode)];
                      var alreadySelected = countryCodes.indexOf(this.selectedCountryData.iso2) !== -1;
                      // check if the given number contains a NANP area code i.e. the only dialCode that could be
                      // extracted was +1 (instead of say +1204) and the actual number's length is >=4
                      var isNanpAreaCode = dialCode === "+1" && numeric.length >= 4;
                      var isRegionlessNanpNumber = this.selectedCountryData.dialCode === "1" && this._isRegionlessNanp(numeric);
                      // only update the flag if:
                      // A) NOT (we currently have a NANP flag selected, and the number is a regionlessNanp)
                      // AND
                      // B) either a matching country is not already selected OR the number contains a NANP area
                      // code (ensure the flag is set to the first matching country)
                      if (!isRegionlessNanpNumber && (!alreadySelected || isNanpAreaCode)) {
                          // if using onlyCountries option, countryCodes[0] may be empty, so we must find the first
                          // non-empty index
                          for (var j = 0; j < countryCodes.length; j++) {
                              if (countryCodes[j]) {
                                  countryCode = countryCodes[j];
                                  break;
                              }
                          }
                      }
                  } else if (number.charAt(0) === "+" && numeric.length) {
                      // invalid dial code, so empty
                      // Note: use getNumeric here because the number has not been formatted yet, so could contain
                      // bad chars
                      countryCode = "";
                  } else if (!number || number === "+") {
                      // empty, or just a plus, so default
                      countryCode = this.defaultCountry;
                  }
                  if (countryCode !== null) {
                      return this._setFlag(countryCode);
                  }
                  return false;
              }
          }, {
              key: "_isRegionlessNanp",
              value: function _isRegionlessNanp(number) {
                  var numeric = this._getNumeric(number);
                  if (numeric.charAt(0) === "1") {
                      var areaCode = numeric.substr(1, 3);
                      return regionlessNanpNumbers.indexOf(areaCode) !== -1;
                  }
                  return false;
              }
          }, {
              key: "_highlightListItem",
              value: function _highlightListItem(listItem) {
                  var prevItem = this.highlightedItem;
                  if (prevItem) prevItem.classList.remove("highlight");
                  this.highlightedItem = listItem;
                  this.highlightedItem.classList.add("highlight");
              }
          }, {
              key: "_getCountryData",
              value: function _getCountryData(countryCode, ignoreOnlyCountriesOption, allowFail) {
                  var countryList = ignoreOnlyCountriesOption ? allCountries : this.countries;
                  for (var i = 0; i < countryList.length; i++) {
                      if (countryList[i].iso2 === countryCode) {
                          return countryList[i];
                      }
                  }
                  if (allowFail) {
                      return null;
                  }
                  throw new Error("No country data for '".concat(countryCode, "'"));
              }
          }, {
              key: "_setFlag",
              value: function _setFlag(countryCode) {
                  var prevCountry = this.selectedCountryData.iso2 ? this.selectedCountryData : {};
                  // do this first as it will throw an error and stop if countryCode is invalid
                  this.selectedCountryData = countryCode ? this._getCountryData(countryCode, false, false) : {};
                  // update the defaultCountry - we only need the iso2 from now on, so just store that
                  if (this.selectedCountryData.iso2) {
                      this.defaultCountry = this.selectedCountryData.iso2;
                  }
                  this.selectedFlagInner.setAttribute("class", "iti-flag ".concat(countryCode));
                  // update the selected country's title attribute
                  var title = countryCode ? "".concat(this.selectedCountryData.name, ": +").concat(this.selectedCountryData.dialCode) : "Unknown";
                  this.selectedFlag.setAttribute("title", title);
                  if (this.options.separateDialCode) {
                      var dialCode = this.selectedCountryData.dialCode ? "+".concat(this.selectedCountryData.dialCode) : "";
                      this.selectedDialCode.innerHTML = dialCode;
                      // add 6px of padding after the grey selected-dial-code box, as this is what we use in the css
                      this.telInput.style.paddingLeft = "".concat(this.selectedFlag.offsetWidth + 6, "px");
                  }
                  // and the input's placeholder
                  this._updatePlaceholder();
                  // update the active list item
                  if (this.options.allowDropdown) {
                      var prevItem = this.activeItem;
                      if (prevItem) {
                          prevItem.classList.remove("active");
                          prevItem.setAttribute("aria-selected", "false");
                      }
                      if (countryCode) {
                          var nextItem = this.countryList.querySelector("#iti-item-".concat(countryCode));
                          nextItem.setAttribute("aria-selected", "true");
                          nextItem.classList.add("active");
                          this.activeItem = nextItem;
                          this.countryList.setAttribute("aria-activedescendant", nextItem.getAttribute("id"));
                      }
                  }
                  // return if the flag has changed or not
                  return prevCountry.iso2 !== countryCode;
              }
          }, {
              key: "_updatePlaceholder",
              value: function _updatePlaceholder() {
                  var shouldSetPlaceholder = this.options.autoPlaceholder === "aggressive" || !this.hadInitialPlaceholder && this.options.autoPlaceholder === "polite";
                  if (window.intlTelInputUtils && shouldSetPlaceholder) {
                      var numberType = intlTelInputUtils.numberType[this.options.placeholderNumberType];
                      var placeholder = this.selectedCountryData.iso2 ? intlTelInputUtils.getExampleNumber(this.selectedCountryData.iso2, this.options.nationalMode, numberType) : "";
                      placeholder = this._beforeSetNumber(placeholder);
                      if (typeof this.options.customPlaceholder === "function") {
                          placeholder = this.options.customPlaceholder(placeholder, this.selectedCountryData);
                      }
                      this.telInput.setAttribute("placeholder", placeholder);
                  }
              }
          }, {
              key: "_selectListItem",
              value: function _selectListItem(listItem) {
                  // update selected flag and active list item
                  var flagChanged = this._setFlag(listItem.getAttribute("data-country-code"));
                  this._closeDropdown();
                  this._updateDialCode(listItem.getAttribute("data-dial-code"), true);
                  // focus the input
                  this.telInput.focus();
                  // put cursor at end - this fix is required for FF and IE11 (with nationalMode=false i.e. auto
                  // inserting dial code), who try to put the cursor at the beginning the first time
                  var len = this.telInput.value.length;
                  this.telInput.setSelectionRange(len, len);
                  if (flagChanged) {
                      this._triggerCountryChange();
                  }
              }
          }, {
              key: "_closeDropdown",
              value: function _closeDropdown() {
                  this.countryList.classList.add("hide");
                  this.countryList.setAttribute("aria-expanded", "false");
                  // update the arrow
                  this.dropdownArrow.classList.remove("up");
                  // unbind key events
                  document.removeEventListener("keydown", this._handleKeydownOnDropdown);
                  document.documentElement.removeEventListener("click", this._handleClickOffToClose);
                  this.countryList.removeEventListener("mouseover", this._handleMouseoverCountryList);
                  this.countryList.removeEventListener("click", this._handleClickCountryList);
                  // remove menu from container
                  if (this.options.dropdownContainer) {
                      if (!this.isMobile) window.removeEventListener("scroll", this._handleWindowScroll);
                      if (this.dropdown.parentNode) this.dropdown.parentNode.removeChild(this.dropdown);
                  }
                  this._trigger("close:countrydropdown");
              }
          }, {
              key: "_scrollTo",
              value: function _scrollTo(element, middle) {
                  var container = this.countryList;
                  // windowTop from https://stackoverflow.com/a/14384091/217866
                  var windowTop = window.pageYOffset || document.documentElement.scrollTop;
                  var containerHeight = container.offsetHeight;
                  var containerTop = container.getBoundingClientRect().top + windowTop;
                  var containerBottom = containerTop + containerHeight;
                  var elementHeight = element.offsetHeight;
                  var elementTop = element.getBoundingClientRect().top + windowTop;
                  var elementBottom = elementTop + elementHeight;
                  var newScrollTop = elementTop - containerTop + container.scrollTop;
                  var middleOffset = containerHeight / 2 - elementHeight / 2;
                  if (elementTop < containerTop) {
                      // scroll up
                      if (middle) newScrollTop -= middleOffset;
                      container.scrollTop = newScrollTop;
                  } else if (elementBottom > containerBottom) {
                      // scroll down
                      if (middle) newScrollTop += middleOffset;
                      var heightDifference = containerHeight - elementHeight;
                      container.scrollTop = newScrollTop - heightDifference;
                  }
              }
          }, {
              key: "_updateDialCode",
              value: function _updateDialCode(newDialCodeBare, hasSelectedListItem) {
                  var inputVal = this.telInput.value;
                  // save having to pass this every time
                  var newDialCode = "+".concat(newDialCodeBare);
                  var newNumber;
                  if (inputVal.charAt(0) === "+") {
                      // there's a plus so we're dealing with a replacement (doesn't matter if nationalMode or not)
                      var prevDialCode = this._getDialCode(inputVal);
                      if (prevDialCode) {
                          // current number contains a valid dial code, so replace it
                          newNumber = inputVal.replace(prevDialCode, newDialCode);
                      } else {
                          // current number contains an invalid dial code, so ditch it
                          // (no way to determine where the invalid dial code ends and the rest of the number begins)
                          newNumber = newDialCode;
                      }
                  } else if (this.options.nationalMode || this.options.separateDialCode) {
                      // don't do anything
                      return;
                  } else {
                      // nationalMode is disabled
                      if (inputVal) {
                          // there is an existing value with no dial code: prefix the new dial code
                          newNumber = newDialCode + inputVal;
                      } else if (hasSelectedListItem || !this.options.autoHideDialCode) {
                          // no existing value and either they've just selected a list item, or autoHideDialCode is
                          // disabled: insert new dial code
                          newNumber = newDialCode;
                      } else {
                          return;
                      }
                  }
                  this.telInput.value = newNumber;
              }
          }, {
              key: "_getDialCode",
              value: function _getDialCode(number) {
                  var dialCode = "";
                  // only interested in international numbers (starting with a plus)
                  if (number.charAt(0) === "+") {
                      var numericChars = "";
                      // iterate over chars
                      for (var i = 0; i < number.length; i++) {
                          var c = number.charAt(i);
                          // if char is number (https://stackoverflow.com/a/8935649/217866)
                          if (!isNaN(parseInt(c, 10))) {
                              numericChars += c;
                              // if current numericChars make a valid dial code
                              if (this.countryCodes[numericChars]) {
                                  // store the actual raw string (useful for matching later)
                                  dialCode = number.substr(0, i + 1);
                              }
                              // longest dial code is 4 chars
                              if (numericChars.length === 4) {
                                  break;
                              }
                          }
                      }
                  }
                  return dialCode;
              }
          }, {
              key: "_getFullNumber",
              value: function _getFullNumber() {
                  var val = this.telInput.value.trim();
                  var dialCode = this.selectedCountryData.dialCode;
                  var prefix;
                  var numericVal = this._getNumeric(val);
                  // normalized means ensure starts with a 1, so we can match against the full dial code
                  var normalizedVal = numericVal.charAt(0) === "1" ? numericVal : "1".concat(numericVal);
                  if (this.options.separateDialCode && val.charAt(0) !== "+") {
                      // when using separateDialCode, it is visible so is effectively part of the typed number
                      prefix = "+".concat(dialCode);
                  } else if (val && val.charAt(0) !== "+" && val.charAt(0) !== "1" && dialCode && dialCode.charAt(0) === "1" && dialCode.length === 4 && dialCode !== normalizedVal.substr(0, 4)) {
                      // ensure national NANP numbers contain the area code
                      prefix = dialCode.substr(1);
                  } else {
                      prefix = "";
                  }
                  return prefix + val;
              }
          }, {
              key: "_beforeSetNumber",
              value: function _beforeSetNumber(originalNumber) {
                  var number = originalNumber;
                  if (this.options.separateDialCode) {
                      var dialCode = this._getDialCode(number);
                      if (dialCode) {
                          // US dialCode is "+1", which is what we want
                          // CA dialCode is "+1 123", which is wrong - should be "+1" (as it has multiple area codes)
                          // AS dialCode is "+1 684", which is what we want (as it doesn't have area codes)
                          // Solution: if the country has area codes, then revert to just the dial code
                          if (this.selectedCountryData.areaCodes !== null) {
                              dialCode = "+".concat(this.selectedCountryData.dialCode);
                          }
                          // a lot of numbers will have a space separating the dial code and the main number, and
                          // some NANP numbers will have a hyphen e.g. +1 684-733-1234 - in both cases we want to get
                          // rid of it
                          // NOTE: don't just trim all non-numerics as may want to preserve an open parenthesis etc
                          var start = number[dialCode.length] === " " || number[dialCode.length] === "-" ? dialCode.length + 1 : dialCode.length;
                          number = number.substr(start);
                      }
                  }
                  return this._cap(number);
              }
          }, {
              key: "_triggerCountryChange",
              value: function _triggerCountryChange() {
                  this._trigger("countrychange");
              }
          }, {
              key: "handleAutoCountry",
              value: function handleAutoCountry() {
                  if (this.options.initialCountry === "auto") {
                      // we must set this even if there is an initial val in the input: in case the initial val is
                      // invalid and they delete it - they should see their auto country
                      this.defaultCountry = window.intlTelInputGlobals.autoCountry;
                      // if there's no initial value in the input, then update the flag
                      if (!this.telInput.value) {
                          this.setCountry(this.defaultCountry);
                      }
                      this.resolveAutoCountryPromise();
                  }
              }
          }, {
              key: "handleUtils",
              value: function handleUtils() {
                  // if the request was successful
                  if (window.intlTelInputUtils) {
                      // if there's an initial value in the input, then format it
                      if (this.telInput.value) {
                          this._updateValFromNumber(this.telInput.value);
                      }
                      this._updatePlaceholder();
                  }
                  this.resolveUtilsScriptPromise();
              }
          }, {
              key: "destroy",
              value: function destroy() {
                  var form = this.telInput.form;
                  if (this.options.allowDropdown) {
                      // make sure the dropdown is closed (and unbind listeners)
                      this._closeDropdown();
                      this.selectedFlag.removeEventListener("click", this._handleClickSelectedFlag);
                      this.flagsContainer.removeEventListener("keydown", this._handleFlagsContainerKeydown);
                      // label click hack
                      var label = this._getClosestLabel();
                      if (label) label.removeEventListener("click", this._handleLabelClick);
                  }
                  // unbind hiddenInput listeners
                  if (this.hiddenInput && form) form.removeEventListener("submit", this._handleHiddenInputSubmit);
                  // unbind autoHideDialCode listeners
                  if (this.options.autoHideDialCode) {
                      if (form) form.removeEventListener("submit", this._handleSubmitOrBlurEvent);
                      this.telInput.removeEventListener("blur", this._handleSubmitOrBlurEvent);
                  }
                  // unbind key events, and cut/paste events
                  this.telInput.removeEventListener("keyup", this._handleKeyupEvent);
                  this.telInput.removeEventListener("cut", this._handleClipboardEvent);
                  this.telInput.removeEventListener("paste", this._handleClipboardEvent);
                  // remove attribute of id instance: data-intl-tel-input-id
                  this.telInput.removeAttribute("data-intl-tel-input-id");
                  // remove markup (but leave the original input)
                  var wrapper = this.telInput.parentNode;
                  wrapper.parentNode.insertBefore(this.telInput, wrapper);
                  wrapper.parentNode.removeChild(wrapper);
                  delete window.intlTelInputGlobals.instances[this.id];
              }
          }, 
              {
              key: "getDialCode",
              value: function getDialCode() {
                  if (window.intlTelInputUtils) {
                      var dialCode = this.selectedCountryData.dialCode;
                      if(dialCode){
                          return "+" + dialCode
                      }
                  }
                  return "";
              }
          }, {
              key: "getDialNumber",
              value: function getDialNumber() {
                  if (getDialNumber) {
                      return getDialNumber;
                  }
                  return "";
              }
          }, {
              key: "getExtension",
              value: function getExtension() {
                  if (window.intlTelInputUtils) {
                      return intlTelInputUtils.getExtension(this._getFullNumber(), this.selectedCountryData.iso2);
                  }
                  return "";
              }
          }, {
              key: "getNumber",
              value: function getNumber(format) {
                  if (window.intlTelInputUtils) {
                      var iso2 = this.selectedCountryData.iso2;
                      return intlTelInputUtils.formatNumber(this._getFullNumber(), iso2, format);
                  }
                  return "";
              }
          }, {
              key: "getNumberType",
              value: function getNumberType() {
                  if (window.intlTelInputUtils) {
                      return intlTelInputUtils.getNumberType(this._getFullNumber(), this.selectedCountryData.iso2);
                  }
                  return -99;
              }
          }, {
              key: "getSelectedCountryData",
              value: function getSelectedCountryData() {
                  return this.selectedCountryData;
              }
          }, {
              key: "getValidationError",
              value: function getValidationError() {
                  if (window.intlTelInputUtils) {
                      var iso2 = this.selectedCountryData.iso2;
                      return intlTelInputUtils.getValidationError(this._getFullNumber(), iso2);
                  }
                  return -99;
              }
          }, {
              key: "isValidNumber",
              value: function isValidNumber() {
                  var val = this._getFullNumber().trim();
                  var countryCode = this.options.nationalMode ? this.selectedCountryData.iso2 : "";
                  return window.intlTelInputUtils ? intlTelInputUtils.isValidNumber(val, countryCode) : null;
              }
          }, {
              key: "setCountry",
              value: function setCountry(originalCountryCode) {
                  var countryCode = originalCountryCode.toLowerCase();
                  // check if already selected
                  if (!this.selectedFlagInner.classList.contains(countryCode)) {
                      this._setFlag(countryCode);
                      this._updateDialCode(this.selectedCountryData.dialCode, false);
                      this._triggerCountryChange();
                  }
              }
          }, {
              key: "setNumber",
              value: function setNumber(number) {
                  // we must update the flag first, which updates this.selectedCountryData, which is used for
                  // formatting the number before displaying it
                  var flagChanged = this._updateFlagFromNumber(number);
                  this._updateValFromNumber(number);
                  if (flagChanged) {
                      this._triggerCountryChange();
                  }
              }
          }, {
              key: "setPlaceholderNumberType",
              value: function setPlaceholderNumberType(type) {
                  this.options.placeholderNumberType = type;
                  this._updatePlaceholder();
              }
          } ]);
          return Iti;
      }();
      /********************
*  STATIC METHODS
********************/
      // get the country data object
      window.intlTelInputGlobals.getCountryData = function() {
          return allCountries;
      };
      // inject a <script> element to load utils.js
      var injectScript = function injectScript(path, handleSuccess, handleFailure) {
          // inject a new script element into the page
          var script = document.createElement("script");
          script.onload = function() {
              forEachInstance("handleUtils");
              if (handleSuccess) handleSuccess();
          };
          script.onerror = function() {
              forEachInstance("rejectUtilsScriptPromise");
              if (handleFailure) handleFailure();
          };
          script.className = "iti-load-utils";
          script.async = true;
          script.src = path;
          document.body.appendChild(script);
      };
      // load the utils script
      window.intlTelInputGlobals.loadUtils = function(path) {
          // 2 options:
          // 1) not already started loading (start)
          // 2) already started loading (do nothing - just wait for the onload callback to fire, which will
          // trigger handleUtils on all instances, invoking their resolveUtilsScriptPromise functions)
          if (!window.intlTelInputUtils && !window.intlTelInputGlobals.startedLoadingUtilsScript) {
              // only do this once
              window.intlTelInputGlobals.startedLoadingUtilsScript = true;
              // if we have promises, then return a promise
              if (typeof Promise !== "undefined") {
                  return new Promise(function(resolve, reject) {
                      return injectScript(path, resolve, reject);
                  });
              }
              injectScript(path);
          }
          return null;
      };
      // default options
      window.intlTelInputGlobals.defaults = defaults;
      // version
      window.intlTelInputGlobals.version = "15.0.0";
      // convenience wrapper
      return function(input, options) {
          var iti = new Iti(input, options);
          iti._init();
          input.setAttribute("data-intl-tel-input-id", iti.id);
          window.intlTelInputGlobals.instances[iti.id] = iti;
          return iti;
      };
  }();
});
document.addEventListener("DOMContentLoaded", function (event) {
  initUrlParams()
  initIntlTelInput()
  initLegacyPasswordValidation()
})

function initUrlParams() {
  var bp = urlParam("bp")
  var cb = urlParam("cb")
  var session = urlParam("session")
  var email = urlParam("email")
  var first_name = urlParam("first_name")
  var last_name = urlParam("last_name")
  var affiliate_id = urlParam("aff_id")
  var domain = urlParam("domain")

  if (affiliate_id) setValueBySelector(".affiliate_id", affiliate_id)
  if (email) {
    setValueBySelector(".emailcb", decodeURIComponent(email))
    setClassBySelector(".emailcb", "valid")
  }

  if (first_name) {
    if (first_name.indexOf("+") == -1) {
      setValueBySelector(".first_nameInclude", decodeURIComponent(first_name))
    } else {
      setValueBySelector(".first_nameInclude", decodeURIComponent(first_name.replace(/\+/g, "%20")))
    }
    setClassBySelector(".first_nameInclude", "valid")
  }
  if (last_name) {
    if (last_name.indexOf("+") == -1) {
      setValueBySelector(".last_nameInclude", decodeURIComponent(last_name))
    } else {
      setValueBySelector(".last_nameInclude", decodeURIComponent(last_name.replace(/\+/g, "%20")))
    }
    setClassBySelector(".last_nameInclude", "valid")
  }
  if (session) {
    setValueBySelector(".session_id", session)
  }
  if (bp == 0) {
    document.querySelectorAll(".hover-modal").forEach((elem) => (elem.style.display = "none"))
  }
  if (cb == 0) {
    exitpage = false
  }
}

function initIntlTelInput() {
  const telCode = document.querySelectorAll(".phone")
  let iti = []

  telCode.forEach((value, indx) => {
    window.intlTelInput(value, {
      allowDropdown: true,
      autoHideDialCode: true,
      autoPlaceholder: "polite",

      formatOnDisplay: true,
      geoIpLookup: function (callback) {
        fetch("https://mamaya.online/geo")
          .then((response) => response.json())
          .then((data) => {
            const countryCode = data && data.country_code ? data.country_code : ""
            callback(countryCode)
          })
          .catch((error) => {
            console.log("Error", error)
            callback("us")
          })
      },
      initialCountry: "auto",
      localizedCountries: {
        ua: "Ukraine",
      },
      nationalMode: true,
      placeholderNumberType: "MOBILE",
      separateDialCode: true,
      utilsScript: "js/utils.js",
    })
    iti[indx] = window.intlTelInputGlobals.getInstance(value)

    value.addEventListener("countrychange", function (elem) {
      if (iti[indx].getDialCode() != "") {
        setValueBySelector(".phonecc", iti[indx].getDialCode())
      }

      iti.forEach(function (instance) {
        var currCountry = iti[indx].getSelectedCountryData()
        instance.setCountry(currCountry.iso2)

        setValueBySelector(".phonecc", currCountry.dialCode)
      })
    })
  })

  const observer = new MutationObserver(function(e) {
    setValueBySelector(".phonecc", e[0].target.innerText)
  });

  observer.observe(document.querySelector(".selected-dial-code"), {characterData: true, childList: true});

  document.querySelectorAll("form input").forEach((input) => {
    input.addEventListener("focus", function () {
      this.parentNode.classList.add("active")
    })
    input.addEventListener("focusout", function () {
      if (this.value !== "") {
        return
      }
      this.parentNode.classList.remove("active")
    })
  })

  function forceNumeric() {
    let input = this
    input.value = input.value.replace(/[^\d -]+/g, "")
  }

  document.querySelectorAll(".phone").forEach((elem) => {
    elem.addEventListener("input", forceNumeric)
  })
  const isChromium = Boolean(window.chrome)
  const mailInputs = document.getElementsByName("email")
  mailInputs.forEach((mailInput) => {
    if (isChromium) {
      mailInput.addEventListener("focus", function (e) {
        this.setAttribute("type", "text")
        this.setAttribute("inputmode", "email")
        this.setAttribute("autocomplete", "email")
      })
      mailInput.addEventListener("blur", function (e) {
        this.setAttribute("type", "email")
      })
    }
    mailInput.addEventListener("input", function (e) {
      this.value = this.value.replaceAll(/[^/[\w-@.]/g, "")
    })
  })
}

function initLegacyPasswordValidation() {
  document.querySelectorAll(".password").forEach((elem) => {
    elem.addEventListener("click", function () {
      document.querySelectorAll(".valid-block").forEach((elem) => (elem.style.display = "block"))
    })
    elem.addEventListener("blur", function () {
      document.querySelectorAll(".valid-block").forEach((elem) => (elem.style.display = "none"))
    })
    elem.addEventListener("input", function () {
      validatePasswordInput(this)
    })
  })

  const validatePasswordInput = (input) => {
    const password = input.value
    let isValid = true

    const validationRules = {
      ".valid-check-A-Z": /[A-Z]/,
      ".valid-check-a-z": /[a-z]/,
      ".valid-check-1-9": /[1-9]/,
      ".valid-check-length": /^.{8,12}$/,
      ".valid-check-alphanumeric": /[A-Za-z0-9]/,
    }

    Object.keys(validationRules).forEach((key) => {
      if (validationRules[key].test(password)) {
        document.querySelectorAll(key).forEach((elem) => {
          if (!elem.classList.contains("check")) elem.classList.add("check")
        })
      } else {
        document.querySelectorAll(key).forEach((elem) => {
          if (elem.classList.contains("check")) elem.classList.remove("check")
        })

        isValid = false
      }
    })

    if (isValid) {
      document.querySelectorAll(".valid-block").forEach((elem) => (elem.style.display = "none"))
    }
  }

  function randString(id) {
    let dataSet = document.querySelector(id).getAttribute("data-character-set").split(",")
    let possible = ""
    if (dataSet.indexOf("0-9") >= 0) {
      possible += "0123456789"
    }
    if (dataSet.indexOf("a-z") >= 0) {
      possible += "abcdefghijklmnopqrstuvwxyz"
    }
    if (dataSet.indexOf("A-Z") >= 0) {
      possible += "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
    }
    let text = ""
    for (let i = 0; i < document.querySelector(id).getAttribute("data-size"); i++) {
      text += possible.charAt(Math.floor(Math.random() * possible.length))
    }
    return text
  }
  document.querySelectorAll('input[rel="gp"]').forEach((elem) => {
    let number = Math.floor(Math.random() * (1 - 9) + 9)
    elem.value = randString(`#${elem.getAttribute("id")}`) + number + "Db"
  })

  document.querySelectorAll(".getNewPass").forEach((elem) => {
    elem.addEventListener("click", function () {
      const number = Math.floor(Math.random() * (1 - 9) + 9)

      document.querySelectorAll('input[rel="gp"]').forEach((elem) => {
        elem.value = randString(`#${elem.getAttribute("id")}`) + number + "Db"
        validatePasswordInput(elem)
      })

      const element = elem.parentNode.parentNode.querySelector(".phone")
      if (element) {
        element.select()
      }
    })
  })
}

const setValueBySelector = (selector, value) => {
  document.querySelectorAll(selector).forEach((elem) => (elem.value = value))
}
const setClassBySelector = (selector, className) => {
  document.querySelectorAll(selector).forEach((elem) => elem.classList.add(className))
}

if (!urlParam) {
  function urlParam(name) {
    var results = new RegExp("[?&]" + name + "=([^&#]*)").exec(window.location.href)
    if (results == null) {
      return null
    } else {
      return results[1] || 0
    }
  }
}




let validMessages
const initValidMessages = () => {
  const formLang = document.querySelector("form").classList[document.querySelector("form").classList.length - 1]
  
  validMessages = globalValidMessages[formLang] ? globalValidMessages[formLang] : globalValidMessages["en"]
}

document.addEventListener("DOMContentLoaded", function () {
  initValidMessages()
  let i = 0
  for (const form of document.querySelectorAll("form")) {
    if (isFormCorrect(form)) {
      initForm(form)
      initFormFields(form)
      initCheckbox(form, i)
      initPreloader(form)
      i++
    }
  }
})

// Regex variables
const regex_require = /\S+/
const regex_email = /^\b[A-Z0-9._%-]+@[A-Z0-9.-]+\.[A-Z]{1,}\b$/i
const regex_username = new RegExp(
  "^([A-Za-z\xAA\xB5\xBA\xC0-\xD6\xD8-\xF6\xF8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u037F\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u052F\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0620-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0840-\u0858\u08A0-\u08B4\u08B6-\u08BD\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971-\u0980\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0AF9\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C39\u0C3D\u0C58-\u0C5A\u0C60\u0C61\u0C80\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0CF1\u0CF2\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D3A\u0D3D\u0D4E\u0D54-\u0D56\u0D5F-\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC-\u0EDF\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8C\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10C7\u10CD\u10D0-\u10FA\u10FC-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F5\u13F8-\u13FD\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u16F1-\u16F8\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u1884\u1887-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191E\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19B0-\u19C9\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1BBA-\u1BE5\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1C80-\u1C88\u1CE9-\u1CEC\u1CEE-\u1CF1\u1CF5\u1CF6\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u209C\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2183\u2184\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2CF2\u2CF3\u2D00-\u2D25\u2D27\u2D2D\u2D30-\u2D67\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005\u3006\u3031-\u3035\u303B\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31BA\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FD5\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA66E\uA67F-\uA69D\uA6A0-\uA6E5\uA717-\uA71F\uA722-\uA788\uA78B-\uA7AE\uA7B0-\uA7B7\uA7F7-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA8FD\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uA9E0-\uA9E4\uA9E6-\uA9EF\uA9FA-\uA9FE\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA7E-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uAAE0-\uAAEA\uAAF2-\uAAF4\uAB01-\uAB06\uAB09-\uAB0E\uAB11-\uAB16\uAB20-\uAB26\uAB28-\uAB2E\uAB30-\uAB5A\uAB5C-\uAB65\uAB70-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC\u0E00-\u0E7F.-]\\s{0,}?)+$"
)

const regex_min_number = (min_number) => new RegExp(".{" + min_number + ",}")
const regex_max_number = (max_number) => new RegExp("^.{1," + max_number + "}$")
const regex_phone = /^\d{7,11}$/
const regex_password = /^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)[a-zA-Z\d]+$/

// Validation rules
const rules = {
  first_name: {
    required: regex_require,
    usernameRegex: regex_username,
    minlength: regex_min_number(2),
    maxlength: regex_max_number(60),
  },
  last_name: {
    required: regex_require,
    usernameRegex: regex_username,
    minlength: regex_min_number(2),
    maxlength: regex_max_number(60),
  },
  password: {
    required: regex_require,
    passwordRegex: regex_password,
    minlength: regex_min_number(8),
    maxlength: regex_max_number(12),
  },
  email: {
    required: regex_require,
    email: regex_email,
  },
  phone: {
    required: regex_require,
    phoneRegex: regex_phone,
  },
}

//Form fields init
const initFormFields = (form) => {
  const phoneccField = stringToHTML(`<input type="hidden" class="phonecc" name="phonecc" value="" />`)
  const sessionIdField = stringToHTML(`<input type="hidden" name="session_id" class="session_id" />`)
  const affiliateIdField = stringToHTML(`<input type="hidden" name="affiliate_id" class="affiliate_id" />`)
  if (usePhoneccField && !form.querySelector(".phonecc")) form.appendChild(phoneccField)
  if (useSessionIdField && !form.querySelector(".session_id")) form.appendChild(sessionIdField)
  if (useAffiliateIdField && !form.querySelector(".affiliate_id")) form.appendChild(affiliateIdField)
}

// Preloader initialization

const initPreloader = (form) => {
  if (form.querySelector(".preloader") || !usePreloader) {
    return
  }

  const preloader = stringToHTML(`<div class="preloader"></div>`)

  form.insertBefore(preloader, form.firstChild)
}

// Checkbox initialization
const initCheckbox = (form, id) => {
  if (form.querySelector("[type=checkbox]") || !useCheckbox) {
    return
  }

  const element = stringToHTML(
    `<div class="checkbox-svg">
      <input type="checkbox" checked id="cbx-${id}" style="display: none" />
      <label for="cbx-${id}" class="checked-svg">
        <svg width="20px" height="20px" viewBox="0 0 18 18">
          <path
            d="M1,9 L1,3.5 C1,2 2,1 3.5,1 L14.5,1 C16,1 17,2 17,3.5 L17,14.5 C17,16 16,17 14.5,17 L3.5,17 C2,17 1,16 1,14.5 L1,9 Z"
          ></path>
          <polyline points="1 9 7 14 15 4"></polyline>
        </svg>
      </label>
      <div class="privacy-checkbox privacy-checkbox_main">
        <p>
          ${validMessages.checkbox.text1} <a href="disclaimers/terms.html" target="_blank">${validMessages.checkbox.link1}</a> 
          ${validMessages.checkbox.text2} <a href="disclaimers/privacy.html" target="_blank">${validMessages.checkbox.link2}</a>
          ${validMessages.checkbox.text3}
        </p>
      </div>
    </div>`
  )

  const submitElement = form.querySelector("[type=submit]")

  if (checkboxAfterSubmit) submitElement.after(element)
  else submitElement.before(element)
}

// Form initialization
const initForm = (form) => {
  form.addEventListener("submit", onSubmit)
  form.addEventListener("focusout", onFocusOut)
  form.addEventListener("keyup", onKeyUp)
}

// Main validate
const validateForm = (form) => {
  if (!isFormCorrect(form)) return null
  let is_valid = true

  for (const rule_field of Object.keys(rules)) {
    const form_field = form.elements[rule_field][0] ? form.elements[rule_field][0] : form.elements[rule_field]
    if (!validateField(form_field)) is_valid = false
  }

  return is_valid
}

// Validate field
const validateField = (form_field) => {
  const field_name = form_field.name

  if (!Object.keys(rules).includes(field_name)) return null

  for (const rule of Object.keys(rules[field_name])) {
    if (!rules[field_name][rule].test(form_field.value)) {
      setErrorField(form_field, validMessages[field_name][rule])
      return false
    } else {
      setValidField(form_field)
    }
  }

  return true
}

// Set field state to error
const setErrorField = (field, error_text) => {
  field.classList.remove("valid")
  field.classList.add("error")

  let errorElement = field.parentNode.querySelector("label.error")
  if (errorElement === null) {
    errorElement = stringToHTML(
      `<label class="error" for="${field.name}" id="${field.name}-error">${error_text}</label>`
    )
    field.parentNode.append(errorElement)
  } else {
    errorElement.innerHTML = error_text
    errorElement.style = "display: flex"
  }
}

// Set field state to valid
const setValidField = (field) => {
  field.classList.remove("error")
  field.classList.add("valid")

  const errorElement = field.parentNode.querySelector("label.error")
  if (errorElement) field.parentNode.removeChild(errorElement)
}

// Check if form have enough needed fields. Return null if not.
const isFormCorrect = (form) => {
  const form__fields = Array.from(form.elements)
  const form__fields_names = form__fields.map((field) => field.name)

  return Object.keys(rules).every((element) => form__fields_names.includes(element))
}

function urlParam(name) {
  var results = new RegExp("[?&]" + name + "=([^&#]*)").exec(window.location.href)
  if (results == null) {
    return null
  } else {
    return results[1] || 0
  }
}

const onKeyUp = (event) => {
  let element = event.target
  validateField(element)
  document.querySelectorAll('[name="' + element.name + '"]').forEach((field) => (field.value = element.value))
}
const onFocusOut = (event) => {
  let element = event.target

  if (element.name === "email") {
    rebuidEmail(element)
  }
  validateField(element)
}

const onSubmit = (event) => {
  event.preventDefault()

  const form = event.currentTarget

  if (!validateForm(form)) return

  document.querySelectorAll(".preloader").forEach((elem) => (elem.style.display = "flex"))

  document.querySelectorAll("input[name='first_name'], input[name='last_name']").forEach((elem) => {
    elem.value = elem.value.substr(0, 60).replace(/[.-]/g, " ").replace(/\s\s+/g, " ")
  })

  const msg = new URLSearchParams(Array.from(new FormData(form))).toString()

  const linkAdress = makeSendAdress()
  console.log("linkAdress= " + linkAdress)

  createXHRRequest(linkAdress, msg)
}

const createXHRRequest = (linkAdress, msg) => {
  const xhr = new XMLHttpRequest()

  xhr.open("POST", linkAdress, true)
  xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded")

  xhr.addEventListener("loadend", () => {
    if (xhr.readyState === XMLHttpRequest.DONE && xhr.status === 200) {
      const data = xhr.response

      const domainForPixel = decodeURIComponent(urlParam("domain")).replace("fbpev", "fbp?ev")
      const obj_data = JSON.parse(data)

      let adress_redir = obj_data.redirect

      if (domainForPixel != null) {
        addPixelElement(domainForPixel)
      }

      if (useThanksPage) {
        setTimeout(function () {
          if (urlParam("noautologin") == 1) {
            adress_redir = "disclaimers/thanks.html"
            localStorage.setItem("redirect", obj_data.redirect)
            window.location = adress_redir
            return true
          }
          adress_redir = "disclaimers/thanks.html"
          localStorage.setItem("redirect", obj_data.redirect)
          window.location = adress_redir
        }, 3000)
        return
      }

      if (urlParam("noautologin") == 1) {
        setTimeout(function () {
          window.location = adress_redir
        }, 3000)
        return true
      }
      setTimeout(function () {
        window.location = adress_redir
      }, 3000)
    } else if (xhr.status === 400) {
      const obj_data = JSON.parse(xhr.responseText)
      for (const key in obj_data.errors) {
        if (key === "CROB") {
          window.location = obj_data.errors[key]
        } else {
          alert(obj_data.errors[key])
        }
      }
      hidePreloader()
    } else {
      alert("Register form field error")
      console.log(xhr)
      hidePreloader()
    }
  })

  xhr.send(msg)
}

function hidePreloader() {
  document.querySelectorAll(".preloader").forEach((elem) => (elem.style.display = "none"))
}

const addPixelElement = (domainForPixel) => {
  const pixelElement = stringToHTML(
    `<iframe width="1" height="1" style="display: none" src="https://${decodeURIComponent(domainForPixel)}"></iframe>`
  )

  document.querySelector("body").prepend(pixelElement)
}

function rebuidEmail(this_element) {
  var tmp_el = this_element.value
  tmp_el = tmp_el
    .replace(/[\.+]{2,}/g, ".")
    .replace(/^\.+/g, "")
    .replace(/\.+$/g, "")
    .replace(/[,\/]/g, ".") // заменяем повторяющиеся точки на одну, убираем точки вначале и в конце, заменяем запятую и слеш на точку

  //=========
  tmp_el = tmp_el.replace(/[.]+\s+com$/g, ".com").replace(/\s+com$/g, ".com") // убираем лишние точки и пробелы перед com
  tmp_el = tmp_el.replace(/[.]+\s+ru$/g, ".ru").replace(/\s+ru$/g, ".ru") // убираем лишние точки и пробелы перед ru
  tmp_el = tmp_el.replace(/[.]+\s+net$/g, ".net").replace(/\s+net$/g, ".net") // убираем лишние точки и пробелы перед net
  tmp_el = tmp_el.replace(/[.]+\s+ua$/g, ".ua").replace(/\s+ua$/g, ".ua") // убираем лишние точки и пробелы перед ua
  //=========

  var brokenDomainsGmail = [
    "gmil",
    "gmaail",
    "gmaij",
    "gmaila",
    "googlemail",
    "jimail",
    "gmailcom",
    "gimailcom",
    "gaiml",
    "gemail",
    "gilmei",
    "gmael",
    "gmaol",
    "gamail",
    "gamil",
    "glail",
    "gmaik",
  ]
  brokenDomainsGmail.forEach((element) => {
    // правка домена gmail
    tmp_el = tmp_el.replace(element, "gmail")
  })

  var brokenDomainsYandex = ["yande[", "jandex"]
  brokenDomainsYandex.forEach((element) => {
    // правка домена yandex
    tmp_el = tmp_el.replace(element, "yandex")
  })

  var brokenDomainsMail = ["email", "meil"]
  brokenDomainsMail.forEach((element) => {
    // правка домена mail.ru
    tmp_el = tmp_el.replace(element, "mail")
  })

  //=========
  tmp_el = tmp_el.replace(/gmail$/g, "gmail.com") // правка на домен первого уровня
  tmp_el = tmp_el.replace(/mail$/g, "mail.ru") // правка на домен первого уровня
  tmp_el = tmp_el.replace(/mail\.ry$/g, "mail.ru") // правка на домен первого уровня
  //=========
  tmp_el = tmp_el.replace(/\s+/g, "").replace(/[/.]{2,}/g, ".") // убираем лишние пробелы и повторяющиеся точки
  tmp_el = tmp_el.replace(/@\s+/g, "@").replace(/\s+@/g, "@") // убираем лишние пробелы до и после собачки
  tmp_el = tmp_el.replace(/[.]+@/g, "@").replace(/@[.]+/g, "@") // убираем лишние точки до и после собачки

  document.querySelectorAll("[name=email]").forEach((elem) => (elem.value = tmp_el)) //вставляем во все инпуты с именем емейл
}

const stringToHTML = (str) => {
  const parser = new DOMParser()
  const doc = parser.parseFromString(str, "text/html")
  return doc.body.children[0]
}
function makeSendAdress() {
  var formOff = window.location.host
  if (formOff.includes("sogefju.com")) {
    alert("test ok")
    return false
  } else {
    var protocol = location.protocol
    var tmp = location.hostname.replace(/[a-z]{2}\./, "")
    tmp = protocol + "//cabinet." + tmp + "/api/register"
    return "send.php"
  }
}
const globalValidMessages = {
  en: {
    first_name: {
      usernameRegex: "Name must be more than 2 characters long, without special characters or spaces",
      required: "The first name field is required",
      minlength: "The first name must be at least 2",
      maxlength: "First name can be a maximum of 25",
    },
    last_name: {
      usernameRegex: "Last name must be more than 2 characters long, without special characters or spaces",
      required: "The last name field is required",
      minlength: "The last name must be at least 2",
      maxlength: "Surname can be a maximum of 25",
    },
    password: {
      passwordRegex:
        'Password must be between 8-12 characters in length, including letters (A-Z, a-z) and numbers (0-9). Without any special symbols (^@()#*+/"?!=.{}~`&) and spaces',
      required: "The password field is required",
      minlength: "The password must be at least 8 characters",
      maxlength: "The password may not be greater than 12 characters",
    },
    email: {
      required: "The e-mail field is required",
      email: "The e-mail must be a valid address",
    },
    phone: {
      phoneRegex: "The phone must be from 7 to 11 characters, without special characters",
      required: "The phone is required",
    },
    checkbox: {
      text1: "By registration I agree and accept Website",
      link1: "terms",
      text2: "of use and",
      link2: "Privacy Policy",
      text3: "",
    },
  },
  ar: {
    first_name: {
      usernameRegex: "يجب أن يتألف الاسم من أكثر من حرفين، وبدون أي رموز خاصة أو مسافات.",
      required: "حقل الاسم الأول إلزامي",
      minlength: "يجب أن يتكون الاسم الأول من حرفين على الأقل.",
      maxlength: "يمكن أن يحتوي الاسم الأول إلى ما يصل 25 حرفًا كحد أقصى.",
    },
    last_name: {
      usernameRegex: "يجب أن يتألف الاسم الأخير من أكثر من حرفين، وبدون أي رموز خاصة أو مسافات.",
      required: "حقل الاسم الأخير إلزامي",
      minlength: "يجب أن يتألف الاسم الأخير من حرفين على الأقل.",
      maxlength: "يمكن أن يحتوي اللقب على ما يصل إلى 25 حرفًا.",
    },
    password: {
      passwordRegex:
        'يجب أن يتراوح طول كلمة المرور من 8 إلى 12 حرف، وتتضمن حروفًا مثل (A-Z، a-z) وأرقامًا (0-9)، وبدون أي رموز خاصة مثل (^@()#*+/"?!=.{}~`&) أو مسافات. ',
      required: "حقل كلمة المرور إلزامي",
      minlength: "يجب أن تتكون كلمة المرور من 8 أحرف على الأقل",
      maxlength: "لا يجوز أن تزيد كلمة المرور عن 12 حرفًا.",
    },
    email: {
      required: "حقل البريد الإلكتروني إلزامي.",
      email: "يجب أن يكون عنوان البريد الإلكتروني صالحًا للاستخدام",
    },
    phone: {
      phoneRegex: "يجب أن يتكون الهاتف من 7 إلى 11 حرفًا، وبدون أي رموز خاصة.",
      required: "رقم الهاتف إلزامي.",
    },
    checkbox: {
      text1: "من خلال التسجيل، فأنا أوافق وأقبل",
      link1: "بشروط استخدام الموقع",
      text2: " و",
      link2: "سياسة الخصوصية.",
      text3: "",
    },
  },
  br: {
    first_name: {
      usernameRegex: "O nome deve ter mais de 2 caracteres, sem caracteres ou espaços especiais",
      required: "O campo de primeiro nome é necessário",
      minlength: "O primeiro nome deve ter pelo menos 2 caracteres",
      maxlength: "O primeiro nome pode ter no máximo 25 caracteres",
    },
    last_name: {
      usernameRegex: "O sobrenome deve ter mais de 2 caracteres, sem caracteres ou espaços especiais",
      required: "O campo de sobrenome é necessário",
      minlength: "O sobrenome deve ter pelo menos 2 caracteres",
      maxlength: "O sobrenome pode ter no máximo 25 caracteres",
    },
    password: {
      passwordRegex:
        'A senha deve ter entre 8 e 12 caracteres, incluindo letras (A-Z, A-Z) e números (0-9). Sem símbolos especiais (^@()#*+/"?! =. {} ~` &) e espaços',
      required: "O campo de senha é necessária",
      minlength: "A senha deve ter pelo menos 8 caracteres",
      maxlength: "A senha pode não ser maior que 12 caracteres",
    },
    email: {
      required: "O campo de e-mail é necessário",
      email: "O e-mail deve ser um endereço válido",
    },
    phone: {
      phoneRegex: "O telefone deve ter de 7 a 11 caracteres, sem caracteres especiais",
      required: "O telefone é necessário",
    },
    checkbox: {
      text1: "Ao me cadastrar, eu concordo e aceito os",
      link1: "termos",
      text2: "de uso e",
      link2: "Política de Privacidade",
      text3: "do site",
    },
  },
  cz: {
    first_name: {
      usernameRegex: "Jméno musí mít více než dva znaky, bez speciálních znaků a mezer.",
      required: "Pole pro křestní jméno je povinné",
      minlength: "Křestní jméno musí mít minimálně 2 znaky",
      maxlength: "Křestní jméno může mít maximálně 25 znaků",
    },
    last_name: {
      usernameRegex: "Příjmení musí mít více než dva znaky, bez speciálních znaků nebo mezer.",
      required: "Pole příjmení je povinné",
      minlength: "Příjmení musí mít alespoň 2 znaky",
      maxlength: "Příjmení může mít maximálně 25 znaků",
    },
    password: {
      passwordRegex:
        'Heslo musí mít délku 8-12 znaků, včetně písmen (A-Z, a-z) a číslic (0-9). Bez speciálních symbolů (^@()#*+/"?!=.{}~`&) a mezer.',
      required: "Pole pro zadání hesla je povinné",
      minlength: "Heslo musí mít alespoň 8 znaků",
      maxlength: "Heslo nesmí mít více než 12 znaků",
    },
    email: {
      required: "Pole e-mail je povinné",
      email: "E-mail musí být platná adresa",
    },
    phone: {
      phoneRegex: "Telefon musí obsahovat 7 až 11 znaků bez speciálních znaků.",
      required: "Telefon je povinný údaj",
    },
    checkbox: {
      text1: "Registrací souhlasím s",
      link1: "Podmínkami a ujednáními",
      text2: "a se",
      link2: "Zásadami ochrany osobních údajů",
      text3: " webových stránek",
    },
  },
  de: {
    first_name: {
      usernameRegex: "Der Name muss mehr als 2 Zeichen lang sein, ohne Sonderzeichen oder Leerzeichen",
      required: "Das Feld für den Vornamen ist erforderlich",
      minlength: "Der Vorname muss aus mindestens 2 Zeichen bestehen",
      maxlength: "Der Vorname kann maximal 25 Zeichen lang sein",
    },
    last_name: {
      usernameRegex: "Der Nachname muss mehr als 2 Zeichen lang sein, ohne Sonderzeichen oder Leerzeichen",
      required: "Das Feld für den Nachnamen ist verpflichtend",
      minlength: "Der Nachname muss aus mindestens 2 Zeichen bestehen",
      maxlength: "Der Nachname darf maximal 25 Zeichen lang sein.",
    },
    password: {
      passwordRegex:
        'Das Passwort muss zwischen 8–12 Zeichen lang sein, einschließlich Buchstaben (A–Z, a–z) und Zahlen (0–9). Ohne Sonderzeichen (^@()#*+/"?!=.{}~`&) und Leerzeichen',
      required: "Das Passwortfeld ist obligatorisch",
      minlength: "Das Passwort muss mindestens 8 Zeichen lang sein",
      maxlength: "Das Passwort darf nicht länger als 12 Zeichen sein",
    },
    email: {
      required: "Das E-Mail-Feld ist erforderlich",
      email: "Die E-Mail muss eine gültige Adresse sein",
    },
    phone: {
      phoneRegex: "Die Telefonnummer muss zwischen 7 und 11 Zeichen lang sein, ohne Sonderzeichen",
      required: "Telefonnummer ist erforderlich",
    },
    checkbox: {
      text1: "Mit der Registrierung akzeptiere ich die",
      link1: "Nutzungsbedingungen",
      text2: "und",
      link2: "Datenschutzerklärung",
      text3: "der Website",
    },
  },
  dk: {
    first_name: {
      usernameRegex: "Navnet skal være længere end 2 tegn, uden specialtegn eller mellemrum.",
      required: "Feltet til fornavn er påkrævet",
      minlength: "Fornavnet skal være mindst 2 tegn",
      maxlength: "Fornavnet kan maksimalt være 25 tegn",
    },
    last_name: {
      usernameRegex: "Efternavnet skal være længere end 2 tegn, uden specialtegn eller mellemrum.",
      required: "Feltet til efternavn er påkrævet",
      minlength: "Efternavnet skal være på mindst 2 tegn",
      maxlength: "Efternavnet kan maksimalt være 25 tegn",
    },
    password: {
      passwordRegex:
        'Adgangskoden skal være mellem 8-12 tegn lang, inklusiv bogstaver (A-Z, a-z) og tal (0-9). Uden specialsymboler (^@()#*+/"?!=.{}~`&) og mellemrum.',
      required: "Feltet med adgangskoden er påkrævet",
      minlength: "Adgangskoden skal være på mindst 8 tegn",
      maxlength: "Adgangskoden må ikke være på mere end 12 tegn",
    },
    email: {
      required: "E-mail feltet er påkrævet",
      email: "E-mailen skal være en gyldig adresse",
    },
    phone: {
      phoneRegex: "Telefonnummeret skal være mellem 7 og 11 tegn, uden specialtegn.",
      required: "Telefonnummeret er påkrævet",
    },
    checkbox: {
      text1: "Ved registrering accepterer jeg og er enig i Websteds",
      link1: "Brugsbetingelser",
      text2: "og",
      link2: "Privatlivspolitik",
      text3: "",
    },
  },
  es: {
    first_name: {
      usernameRegex: "El nombre debe tener más de 2 caracteres, sin caracteres especiales ni espacios",
      required: "El campo de nombre es obligatorio",
      minlength: "El primer nombre debe ser de al menos 2 caracteres",
      maxlength: "El nombre puede ser un máximo de 25 caracteres",
    },
    last_name: {
      usernameRegex: "El apellido debe tener más de 2 caracteres, sin caracteres especiales ni espacios",
      required: "El campo del apellido es obligatorio",
      minlength: "El apellido debe ser de al menos 2 caracteres",
      maxlength: "El apellido puede ser un máximo de 25 caracteres",
    },
    password: {
      passwordRegex:
        'La contraseña debe tener entre 8 y 12 caracteres de longitud, incluidas letras (AZ, az) y números (0-9). Sin símbolos especiales (^@()#*+/"?!=.{}~`&) ni espacios',
      required: "El campo de contraseña es obligatorio",
      minlength: "La contraseña debe tener al menos 8 caracteres",
      maxlength: "La contraseña no puede tener más de 12 caracteres",
    },
    email: {
      required: "El campo de correo electrónico es obligatorio",
      email: "El correo electrónico debe ser una dirección válida",
    },
    phone: {
      phoneRegex: "El teléfono debe tener de 7 a 11 caracteres, sin caracteres especiales",
      required: "El teléfono es requerido",
    },
    checkbox: {
      text1: "Al registrarme, acepto y estoy de acuerdo con",
      link1: "los términos",
      text2: " de uso y",
      link2: "la Política de privacidad",
      text3: "del sitio web",
    },
  },
  fi: {
    first_name: {
      usernameRegex: "Etunimen tulee olla vähintään 2 merkkiä pitkä, erityismerkkejä tai välilyöntejä ei sallita",
      required: "Etunimi vaaditaan",
      minlength: "Etunimen tulee sisältää vähintään 2 merkkiä",
      maxlength: "Etunimi voi sisältää enintään 25 merkkiä",
    },
    last_name: {
      usernameRegex: "Sukunimen tulee olla vähintään 2 merkkiä pitkä, erityismerkkejä tai välilyöntejä ei sallita",
      required: "Sukunimi vaaditaan",
      minlength: "Sukunimen tulee sisältää vähintään 2 merkkiä",
      maxlength: "Sukunimi voi sisältää enintään 25 merkkiä",
    },
    password: {
      passwordRegex:
        'Salasanan tulee olla 8-12 merkkiä pitkä, sisältää kirjaimia (A-Z, a-z) ja numeroita (0-9). Erikoismerkkejä (^@()#*+/"?!=.{}~`&) ja välilyöntejä ei sallita.',
      required: "Salasana vaaditaan",
      minlength: "Salasanan tulee sisältää vähintään 8 merkkiä",
      maxlength: "Salasana voi sisältää enintään 12 merkkiä",
    },
    email: {
      required: "Sähköposti vaaditaan",
      email: "Sähköpostin tulee olla voimassa",
    },
    phone: {
      phoneRegex: "Puhelinnumeron tulee sisältää 7-11 merkkiä, erikoismerkkejä ei sallita",
      required: "Puhelinnumero vaaditaan",
    },
    checkbox: {
      text1: "Rekisteröitymällä hyväksyn verkkosivuston",
      link1: "käyttöehdot",
      text2: "ja",
      link2: "tietosuojakäytännön",
      text3: "",
    },
  },
  fr: {
    first_name: {
      usernameRegex: "Le prénom doit avoir plus de 2 caractères, sans caractères spéciaux ni espaces",
      required: "Le champ du prénom est obligatoire",
      minlength: "Le prénom doit en avoir au moins 2 caractères",
      maxlength: "Le prénom peut avoir un maximum de 25 caractères",
    },
    last_name: {
      usernameRegex: "Le nom doit avoir plus de 2 caractères, sans caractères spéciaux ni espaces",
      required: "Le champ du nom est obligatoire",
      minlength: "Le nom doit en avoir au moins 2 caractères",
      maxlength: "Le nom peut avoir un maximum de 25 caractères",
    },
    password: {
      passwordRegex:
        'Le mot de passe doit avoir entre 8 et 12 caractères, comprenant des lettres (A-Z, a-z) et des chiffres (0-9). Il ne doit pas contenir de symboles spéciaux (^@()#*+/"?!=.{}~`&) ni d`espaces.',
      required: "Le champ du mot de passe est obligatoire",
      minlength: "Le mot de passe doit avoir au moins 8 caractères",
      maxlength: "Le mot de passe ne doit pas avoir plus de 12 caractères",
    },
    email: {
      required: "Le champ de l'email est obligatoire",
      email: "L'email doit être une adresse valide",
    },
    phone: {
      phoneRegex: "Le numéro de téléphone doit avoir entre 7 et 11 caractères, sans caractères spéciaux",
      required: "Le téléphone est obligatoire",
    },
    checkbox: {
      text1: "En vous inscrivant, vous acceptez",
      link1: "les conditions",
      text2: " d'utilisation et la",
      link2: "politique de confidentialité",
      text3: "du sit",
    },
  },
  gr: {
    first_name: {
      usernameRegex: "Το όνομα πρέπει να έχει μήκος πάνω από 2 χαρακτήρες, χωρίς ειδικούς χαρακτήρες ή κενά",
      required: "Το πεδίο του ονόματος είναι υποχρεωτικό",
      minlength: "Το όνομα πρέπει να είναι τουλάχιστον 2 χαρακτήρες",
      maxlength: "Το όνομα μπορεί να είναι το πολύ 25 χαρακτήρες",
    },
    last_name: {
      usernameRegex: "Το επώνυμο πρέπει να έχει μήκος μεγαλύτερο από 2 χαρακτήρες, χωρίς ειδικούς χαρακτήρες ή κενά",
      required: "Το πεδίο του επωνύμου είναι υποχρεωτικό",
      minlength: "Το επώνυμο πρέπει να είναι τουλάχιστον 2 χαρακτήρες",
      maxlength: "Το επώνυμο μπορεί να είναι το πολύ 25 χαρακτήρες",
    },
    password: {
      passwordRegex:
        'Ο κωδικός πρόσβασης πρέπει να έχει μήκος μεταξύ 8-12 χαρακτήρων, συμπεριλαμβανομένων γραμμάτων (A-Z, a-z) και αριθμών (0-9). Χωρίς ειδικά σύμβολα (^@()#*+/"?!=.{}~`&) και κενά.',
      required: "Το πεδίο του κωδικού πρόσβασης είναι υποχρεωτικό",
      minlength: "Ο κωδικός πρόσβασης πρέπει να αποτελείται από τουλάχιστον 8 χαρακτήρες",
      maxlength: "Ο κωδικός πρόσβασης δεν μπορεί να είναι μεγαλύτερος από 12 χαρακτήρες",
    },
    email: {
      required: "Το πεδίο του e-mail είναι υποχρεωτικό",
      email: "Το e-mail πρέπει να είναι έγκυρη διεύθυνση",
    },
    phone: {
      phoneRegex: "Το τηλέφωνο πρέπει να αποτελείται από 7 έως 11 χαρακτήρες, χωρίς ειδικούς χαρακτήρες",
      required: "Το τηλέφωνο είναι υποχρεωτικό",
    },
    checkbox: {
      text1: "Με την εγγραφή μου, συμφωνώ με και αποδέχομαι",
      link1: "τους όρους",
      text2: "χρήσης της Ιστοσελίδας και την",
      link2: "Πολιτική Απορρήτου",
      text3: "",
    },
  },
  hu: {
    first_name: {
      usernameRegex: "A névnek több mint 2 karakter hosszúnak kell lennie, különleges karakterek és szóközök nélkül.",
      required: "A keresztnév mező kitöltése kötelező",
      minlength: "A keresztnévnek legalább 2 karakterből kell állnia",
      maxlength: "A keresztnév legfeljebb 25 karakter lehet",
    },
    last_name: {
      usernameRegex:
        "A vezetéknévnek több mint 2 karakter hosszúnak kell lennie, különleges karakterek és szóközök nélkül.",
      required: "A vezetéknév mező kitöltése kötelező",
      minlength: "A vezetéknévnek legalább 2 karakterből kell állnia",
      maxlength: "A vezetéknév legfeljebb 25 karakter lehet",
    },
    password: {
      passwordRegex:
        'A jelszónak 8-12 karakter hosszúságúnak kell lennie, beleértve a betűket (A-Z, a-z) és a számokat (0-9). Különleges szimbólumok (^@()#*+/"?!=.{}~`&) és szóközök nélkül.',
      required: "A jelszó mező kitöltése kötelező",
      minlength: "A jelszónak legalább 8 karakterből kell állnia",
      maxlength: "A jelszó nem lehet 12 karakternél hosszabb",
    },
    email: {
      required: "Az e-mail mező kitöltése kötelező",
      email: "Az e-mail címnek érvényesnek kell lennie",
    },
    phone: {
      phoneRegex: "A telefonszámnak 7 és 11 karakter között kell lennie, különleges karakterek nélkül.",
      required: "A telefonszám megadása kötelező",
    },
    checkbox: {
      text1: "Regisztrációval elfogadom és elfogadom a weboldal",
      link1: "használati feltételeit",
      text2: "és az",
      link2: " adatvédelmi irányelveket",
      text3: "",
    },
  },
  it: {
    first_name: {
      usernameRegex: "Il nome deve essere lungo più di 2 caratteri, senza caratteri speciali o spazi.",
      required: "Il campo del nome è obbligatorio",
      minlength: "Il nome deve essere almeno 2 caratteri",
      maxlength: "Il nome può essere al massimo 25 caratteri",
    },
    last_name: {
      usernameRegex: "Il cognome deve essere composto da più di 2 caratteri, senza caratteri speciali o spazi.",
      required: "Il campo del cognome è obbligatorio",
      minlength: "Il cognome deve essere almeno 2 caratteri",
      maxlength: "Il cognome può essere al massimo 25 caratteri",
    },
    password: {
      passwordRegex:
        'La password deve avere una lunghezza compresa tra 8 e 12 caratteri, tra cui lettere (A-Z, a-z) e numeri (0-9). Senza simboli speciali (^@()#*+/"?!=.{}~`&) e spazi.',
      required: "Il campo della password è obbligatorio",
      minlength: "La password deve essere composta da almeno 8 caratteri",
      maxlength: "La password non può essere superiore a 12 caratteri",
    },
    email: {
      required: "Il campo e-mail è obbligatorio",
      email: "L'e-mail deve essere un indirizzo valido",
    },
    phone: {
      phoneRegex: "Il telefono deve essere composto da 7 a 11 caratteri, senza caratteri speciali.",
      required: "Il telefono è richiesto",
    },
    checkbox: {
      text1: "Registrandomi, accetto i",
      link1: "Termini e Condizioni",
      text2: " di utilizzo del sito e",
      link2: "l'informativa sulla privacy",
      text3: " del sito web.",
    },
  },
  latam: {
    first_name: {
      usernameRegex: "El nombre debe tener más de 2 caracteres, sin caracteres especiales ni espacios.",
      required: "El campo del primer nombre es obligatorio",
      minlength: "El primer nombre debe tener al menos 2 caracteres",
      maxlength: "El primer nombre puede tener un máximo de 25 caracteres",
    },
    last_name: {
      usernameRegex: "El apellido debe tener más de 2 caracteres, sin caracteres especiales ni espacios.",
      required: "El campo del apellido es obligatorio",
      minlength: "El apellido debe tener al menos 2 caracteres",
      maxlength: "El apellido puede tener un máximo de 25 caracteres",
    },
    password: {
      passwordRegex:
        'La contraseña debe tener entre 8 y 12 caracteres, incluyendo letras (A-Z, a-z) y números (0-9). Sin símbolos especiales (^@()#*+/"?!=.{}~`&) ni espacios.',
      required: "El campo de contraseña es obligatorio",
      minlength: "La contraseña debe tener al menos 8 caracteres",
      maxlength: "La contraseña no puede tener más de 12 caracteres",
    },
    email: {
      required: "El campo de correo electrónico es obligatorio",
      email: "El correo electrónico debe ser una dirección válida",
    },
    phone: {
      phoneRegex: "El teléfono debe tener de 7 a 11 caracteres, sin caracteres especiales",
      required: "El teléfono es obligatorio",
    },
    checkbox: {
      text1: "Al registrarme, acepto y estoy de acuerdo con",
      link1: "los términos",
      text2: "de uso y",
      link2: "la Política de privacidad",
      text3: "del sitio web",
    },
  },
  me: {
    first_name: {
      usernameRegex: "Ime mora biti duže od dva znaka, i ne smije sadržati specijalne znakove ili razmake",
      required: "Polje sa imenom mora biti popunjeno",
      minlength: "Ime mora biti duže od dva znaka",
      maxlength: "Ime ne može biti duže od 25 znakova",
    },
    last_name: {
      usernameRegex: "Prezime mora biti duže od dva znaka, i ne smije sadržati specijalne znakove ili razmake",
      required: "Polje sa prezimenom mora biti popunjeno",
      minlength: "Prezime mora biti duže od dva znaka",
      maxlength: "Prezime ne može biti duže od 25 znakova",
    },
    password: {
      passwordRegex:
        'Lozinka mora sadržati između 8 i 12 znakova, i mora uključivati slova (A -Z, a -z) i brojeve (0-9), i ne smije sadržati specijalne znakove (^@()#*+/"?!=.{}~`&) i razmake',
      required: "Polje sa lozinkom mora biti popunjeno",
      minlength: "Lozinka mora sadržati najmanje 8 znakova",
      maxlength: "Lozinka ne smije biti duža od 12 znakova",
    },
    email: {
      required: "Polje za imejl adresu mora biti popunjeno",
      email: "Imejl adresa mora biti validna",
    },
    phone: {
      phoneRegex: "Telefonski broj mora sadržati između 7 i 11 znakova, i ne smije sadržati specijalne znakove",
      required: "Polje sa telefonskim brojem mora biti popunjeno",
    },
    checkbox: {
      text1: "Registracijom se slažem i prihvaćam",
      link1: "uvjete",
      text2: "korištenja web stranice i",
      link2: "Pravila o privatnosti",
      text3: "",
    },
  },
  nl: {
    first_name: {
      usernameRegex: "Naam moet langer zijn dan 2 tekens, zonder speciale tekens of spaties.",
      required: "Het veld voornaam is verplicht.",
      minlength: "De voornaam moet minimaal 2 tekens lang zijn.",
      maxlength: "De voornaam mag maximaal 25 tekens bevatten.",
    },
    last_name: {
      usernameRegex: "Achternaam moet langer zijn dan 2 tekens, zonder speciale tekens of spaties.",
      required: "Het veld achternaam is verplicht.",
      minlength: "De achternaam moet minimaal 2 tekens lang zijn.",
      maxlength: "De achternaam mag maximaal 25 tekens bevatten.",
    },
    password: {
      passwordRegex:
        'Het wachtwoord moet tussen de 8 en 12 tekens lang zijn, inclusief letters (A-Z, a-z) en cijfers (0-9). Zonder speciale tekens (^@()#*+/"?!=.{}~`&) en spaties.',
      required: "Het veld wachtwoord is verplicht.",
      minlength: "Het wachtwoord moet minimaal 8 tekens lang zijn.",
      maxlength: "Het wachtwoord mag niet langer zijn dan 12 tekens.",
    },
    email: {
      required: "Het veld e-mail is verplicht.",
      email: "Het e-mailadres moet een geldig adres zijn.",
    },
    phone: {
      phoneRegex: "Het telefoonnummer moet tussen de 7 en 11 tekens lang zijn, zonder speciale tekens.",
      required: "Het telefoon veld is verplicht.",
    },
    checkbox: {
      text1: "Door te registreren ga ik akkoord en accepteer ik de algemene",
      link1: "voorwaarden",
      text2: " en het",
      link2: "privacybeleid",
      text3: "van de website",
    },
  },
  no: {
    first_name: {
      usernameRegex: "Navnet må bestå av mer enn 2 tegn, uten spesialtegn eller mellomrom",
      required: "Fornavn-feltet er obligatorisk",
      minlength: "Fornavnet må være minst 2 tegn",
      maxlength: "Fornavn kan være maksimalt 25 tegn",
    },
    last_name: {
      usernameRegex: "Etternavnet må bestå av mer enn 2 tegn, uten spesialtegn eller mellomrom",
      required: "Etternavn-feltet er obligatorisk",
      minlength: "Etternavnet må være minst 2 tegn",
      maxlength: "Etternavn kan være maksimalt 25 tegn",
    },
    password: {
      passwordRegex:
        'Passordet må bestå av mellom 8 og 12 tegn, inkludert bokstaver (A-Z, a-z) og tall (0-9). Uten noen spesielle symboler (^@()#*+/"?!=.{}~`&) eller mellomrom',
      required: "Passord-feltet er obligatorisk",
      minlength: "Passordet må være minst 8 tegn",
      maxlength: "Passordet kan ikke være mer enn 12 tegn",
    },
    email: {
      required: "E-post-feltet er obligatorisk",
      email: "E-postenadressen må være gyldig",
    },
    phone: {
      phoneRegex: "Telefonnummer må være mellom 7 og 11 tegn, uten spesialtegn",
      required: "Telefonnummeret er obligatorisk",
    },
    checkbox: {
      text1: "Ved registrering godtar jeg og aksepterer jeg Nettsidens",
      link1: "vilkår for bruk",
      text2: "og",
      link2: "personvernregler",
      text3: "",
    },
  },
  pl: {
    first_name: {
      usernameRegex: "Imię musi składać się z więcej niż 2 znaków, bez znaków specjalnych i spacji.",
      required: "Pole imienia jest wymagane",
      minlength: "Imię musi składać się z przynajmniej 2 znaków",
      maxlength: "Imię może zawierać maksymalnie 25 liter",
    },
    last_name: {
      usernameRegex: "Nazwisko musi składać się z więcej niż 2 znaków, bez znaków specjalnych i spacji.",
      required: "Pole nazwiska jest wymagane",
      minlength: "Nazwisko musi zawierać co najmniej 2 znaki",
      maxlength: "Nazwisko może zawierać maksymalnie 25 liter",
    },
    password: {
      passwordRegex:
        'Hasło musi zawierać od 8 do 12 znaków, w tym litery (A-Z, a-z) i cyfry (0-9). Bez symboli specjalnych (^@()#*+/"?!=.{}~`&) i spacji.',
      required: "Pole hasła jest wymagane",
      minlength: "Hasło musi składać się z przynajmniej 8 znaków",
      maxlength: "Hasło nie może być dłuższe niż 12 znaków",
    },
    email: {
      required: "Pole adresu e-mail jest wymagane",
      email: "Adres e-mail musi być właściwy",
    },
    phone: {
      phoneRegex: "Numer musi składać się od 7 do 11 znaków bez znaków specjalnych",
      required: "Numer telefonu jest wymagany",
    },
    checkbox: {
      text1: "Rejestrując się, akceptuję i zgadzam się na",
      link1: "warunki",
      text2: "korzystania z niniejszej strony internetowej oraz jej",
      link2: "politykę prywatności",
      text3: "",
    },
  },
  pt: {
    first_name: {
      usernameRegex: "O nome tem de ter mais de 2 caracteres, sem caracteres especiais ou espaços",
      required: "O campo do nome é obrigatório",
      minlength: "O nome deve ter pelo menos 2 caracteres",
      maxlength: "O nome pode ter no máximo 25 caracteres",
    },
    last_name: {
      usernameRegex: "O apelido tem de ter mais de 2 caracteres, sem caracteres especiais ou espaços",
      required: "O campo do apelido é obrigatório",
      minlength: "O apelido deve ter pelo menos 2 caracteres",
      maxlength: "O apelido pode ter no máximo 25 caracteres",
    },
    password: {
      passwordRegex:
        'A palavra-passe deve ter entre 8 e 12 caracteres, incluindo letras (A-Z, a-z) e números (0-9). Não deve incluir quaisquer símbolos especiais (^@()#*+/"?!=.{}~`&) e espaços',
      required: "O campo da palavra-passe é obrigatório",
      minlength: "A palavra-passe deve ter pelo menos 8 caracteres",
      maxlength: "A palavra-passe não pode ter mais de 12 caracteres",
    },
    email: {
      required: "O campo de e-mail é obrigatório",
      email: "O e-mail deve ser um endereço válido",
    },
    phone: {
      phoneRegex: "O telefone deve ter entre 7 e 11 caracteres, sem caracteres especiais",
      required: "O telefone é obrigatório",
    },
    checkbox: {
      text1: "Ao me cadastrar, eu concordo e aceito os",
      link1: "termos",
      text2: "de uso e",
      link2: "Política de Privacidade",
      text3: "do site",
    },
  },
  ro: {
    first_name: {
      usernameRegex: "Numele trebuie să aibă mai mult de 2 caractere, și să nu aibă caractere speciale sau spații.",
      required: "Câmpul prenumele este obligatoriu",
      minlength: "Prenumele trebuie să aibă cel puțin 2 caractere",
      maxlength: "Prenumele poate avea maximum 25 caractere",
    },
    last_name: {
      usernameRegex:
        "Numele de familie trebuie să aibă mai mult de 2 caractere, și să nu aibă caractere speciale sau spații.",
      required: "Câmpul nume de familie este obligatoriu",
      minlength: "Numele de familie trebuie să aibă cel puțin 2 caractere",
      maxlength: "Numele de familie poate avea maximum 25 caractere",
    },
    password: {
      passwordRegex:
        'Parola trebuie să aibă o lungime cuprinsă între 8-12 caractere, inclusiv litere (A-Z, a-z) și numere (0-9). Fără simboluri speciale (^@()#*+/"?!=.{}~`&) și spații.',
      required: "Câmpul parola este obligatoriu",
      minlength: "Parola trebuie să fie de cel puțin 8 caractere",
      maxlength: "Parola nu poate fi mai mare de 12 caractere",
    },
    email: {
      required: "Câmpul e-mail este obligatoriu",
      email: "E-mailul trebuie să fie o adresă validă",
    },
    phone: {
      phoneRegex: "Telefonul trebuie să aibă între 7 și 11 caractere, fără caractere speciale.",
      required: "Telefonul este obligatoriu",
    },
    checkbox: {
      text1: "Prin înregistrare sunt de acord și accept",
      link1: "condițiile",
      text2: " de utilizare ale site-ului web și",
      link2: "politica acestuia de confidențialitate",
      text3: "",
    },
  },
  ru: {
    first_name: {
      usernameRegex: "Имя должно состоять более чем из 2 знаков, без специальных символов и пробелов",
      required: "Поле обязательно для заполнения",
      minlength: "Имя должно состоять не менее чем из 2 знаков",
      maxlength: "Имя может содержать максимум 25 знаков",
    },
    last_name: {
      usernameRegex: "Фамилия должна состоять более чем из 2 знаков, без специальных символов и пробелов",
      required: "Поле обязательно для заполнения",
      minlength: "Фамилия должна состоять не менее чем из 2 знаков",
      maxlength: "Фамилия может содержать максимум 25 знаков",
    },
    password: {
      passwordRegex:
        'Пароль должен состоять из 8-12 знаков, включая буквы (A-Z, a-z) и цифры (0-9). Без специальных символов (^@()#*+/"?!=.{}~`&) и пробелов.',
      required: "Поле обязательно для заполнения",
      minlength: "Пароль должен состоять не менее чем из 8 знаков",
      maxlength: "Пароль не может быть больше 12 знаков",
    },
    email: {
      required: "Поле обязательно для заполнения",
      email: "Адрес электронной почты должен быть действительным",
    },
    phone: {
      phoneRegex: "Телефон должен состоять из 7-11 знаков, без специальных символов",
      required: "Поле обязательно для заполнения",
    },
    checkbox: {
      text1: "Регистрируясь, я соглашаюсь и принимаю",
      link1: "условия",
      text2: " использования веб-сайта и",
      link2: "политику конфиденциальности",
      text3: "",
    },
  },
  se: {
    first_name: {
      usernameRegex: "Namnet måste vara mer än 2 tecken långt, utan specialtecken eller mellanslag",
      required: "Fältet för förnamn är obligatoriskt",
      minlength: "Förnamnet måste vara minst 2 tecken",
      maxlength: "Förnamnet kan vara högst 25 tecken",
    },
    last_name: {
      usernameRegex: "Efternamnet måste vara mer än 2 tecken långt, utan specialtecken eller mellanslag",
      required: "Fältet för efternamn är obligatoriskt",
      minlength: "Efternamnet måste vara minst 2 tecken",
      maxlength: "Efternamnet kan vara högst 25 tecken",
    },
    password: {
      passwordRegex:
        'Lösenordet måste bestå av 8-12 tecken, inklusive bokstäver (A-Z, a-z) och siffror (0-9). Utan specialsymboler (^@()#*+/"?!=.{}~`&) och mellanslag',
      required: "Fältet för lösenord är obligatoriskt",
      minlength: "Lösenordet måste bestå av minst 8 tecken",
      maxlength: "Lösenordet får inte vara längre än 12 tecken",
    },
    email: {
      required: "E-postfältet är obligatoriskt",
      email: "E-postadressen måste vara en giltig adress",
    },
    phone: {
      phoneRegex: "Telefonnumret måste vara mellan 7 och 11 tecken, utan specialtecken",
      required: "Telefonnummer krävs",
    },
    checkbox: {
      text1: "Genom att registrera mig godkänner och accepterar jag webbplatsens",
      link1: "användarvillkor",
      text2: "och",
      link2: "integritetspolicy",
      text3: "",
    },
  },
  sk: {
    first_name: {
      usernameRegex: "Meno musí byť dlhšie ako 2 znaky, bez špeciálnych znakov alebo medzier",
      required: "Pole s menom je povinné",
      minlength: "Meno musí obsahovať aspoň 2 znaky",
      maxlength: "Meno môže byť maximálne 25 znakov dlhé",
    },
    last_name: {
      usernameRegex: "Priezvisko musí byť dlhšie ako 2 znaky, bez špeciálnych znakov alebo medzier",
      required: "Pole pre priezvisko je povinné",
      minlength: "Priezvisko musí obsahovať aspoň 2 znaky",
      maxlength: "Priezvisko môže byť maximálne 25 znakov dlhé",
    },
    password: {
      passwordRegex:
        'Heslo musí mať dĺžku 8 až 12 znakov vrátane písmen (A-Z, a-z) a číslic (0-9). Bez špeciálnych symbolov (^@()#*+/"?!=.{}~`&) a medzier',
      required: "Pole pre heslo je povinné",
      minlength: "Heslo musí obsahovať aspoň 8 znakov",
      maxlength: "Heslo nesmie mať viac ako 12 znakov",
    },
    email: {
      required: "Pole pre e-mail je povinné",
      email: "E-mail musí byť platná adresa",
    },
    phone: {
      phoneRegex: "Telefónne číslo musí mať 7 až 11 znakov bez špeciálnych znakov",
      required: "Telefónne číslo je povinné",
    },
    checkbox: {
      text1: "Registráciou súhlasím a prijímam",
      link1: "podmienky",
      text2: "používania webových stránok",
      link2: "a zásady ochrany osobných údajov",
      text3: "",
    },
  },
  sl: {
    first_name: {
      usernameRegex: "Ime mora biti dolgo več kot 2 znaka, pri tem pa ne sme vsebovati posebnih znakov ali presledkov",
      required: "Polje z imenom je obvezno",
      minlength: "Ime mora imeti vsaj 2 znaka",
      maxlength: "Ime ima lahko največ 25 znakov",
    },
    last_name: {
      usernameRegex:
        "Priimek mora biti dolg več kot 2 znaka, pri tem pa ne sme vsebovati posebnih znakov ali presledkov",
      required: "Polje s priimkom je obvezno",
      minlength: "Priimek mora imeti vsaj 2 znaka",
      maxlength: "Priimek ima lahko največ 25 znakov",
    },
    password: {
      passwordRegex:
        'Geslo mora biti dolgo med 8 in 12 znaki, vključno s črkami (A-Z, a-z) in številkami (0-9). Ne sme pa vsebovati posebnih simbolov (^@()#*+/"?!=.{}~`&) in presledkov',
      required: "Polje z geslom je obvezno",
      minlength: "Geslo mora imeti vsaj 8 znakov",
      maxlength: "Geslo ima lahko največ 12 znakov",
    },
    email: {
      required: "Polje z elektronsko pošto je obvezno",
      email: "Elektronski naslov mora biti veljaven",
    },
    phone: {
      phoneRegex: "Številka telefona mora biti dolga med 7 in 11 znakov, pri tem pa ne sme vsebovati posebnih znakov",
      required: "Telefon je obvezen",
    },
    checkbox: {
      text1: "Z registracijo se strinjam in sprejemam",
      link1: "pogoje",
      text2: "uporabe spletnega mesta in",
      link2: "pravilnik o zasebnosti",
      text3: "",
    },
  },
  tr: {
    first_name: {
      usernameRegex: "İsim 2 karakterden uzun olmalı, özel karakter veya boşluk içermemelidir",
      required: "İsim alanı boş bırakılamaz",
      minlength: "İsim en az 2 karakter olmalıdır",
      maxlength: "İsim maksimum 25 karakter olabilir",
    },
    last_name: {
      usernameRegex: "Soyisim 2 karakterden uzun olmalı, özel karakter veya boşluk içermemelidir",
      required: "Soyisim alanı boş bırakılamaz",
      minlength: "Soyisim en az 2 karakter olmalıdır",
      maxlength: "Soyisim maksimum 25 karakter olabilir",
    },
    password: {
      passwordRegex:
        'Şifre 8-12 karakter uzunluğunda olmalı, harf (A-Z, a-z) ve sayı (0-9) içermelidir. Herhangi bir özel sembol (^@()#*+/"?!=.{}~`&) ve boşluk içermemelidir',
      required: "Şifre alanı boş bırakılamaz",
      minlength: "Şifre en az 8 karakter olmalıdır",
      maxlength: "Şifre 12 karakterden fazla olamaz",
    },
    email: {
      required: "E-posta alanı boş bırakılamaz",
      email: "E-posta geçerli bir adres olmalıdır",
    },
    phone: {
      phoneRegex: "Telefon numarası 7 ila 11 karakter arasında olmalı, özel karakter içermelidir",
      required: "Telefon numarası boş bırakılamaz",
    },
    checkbox: {
      text1: "Kaydolarak Web",
      link1: "Sitesi kullanım şartlarını",
      text2: "ve",
      link2: "Gizlilik Politikasını",
      text3: "kabul ediyor ve onaylıyorum",
    },
  },
  lv: {
    first_name: {
      usernameRegex: "Vārdam jābūt garākam par 2 rakstzīmēm, bez īpašajām rakstzīmēm un atstarpēm",
      required: "Lauks Vārds ir obligāts",
      minlength: "Vārdam jābūt vismaz 2 rakstzīmēm garam",
      maxlength: "Vārds nedrīkst būt garāks par 25 rakstzīmēm",
    },
    last_name: {
      usernameRegex: "Uzvārdam jābūt garākam par 2 rakstzīmēm, bez īpašajām rakstzīmēm un atstarpēm",
      required: "Lauks Uzvārds ir obligāts",
      minlength: "Uzvārdam jābūt vismaz 2 rakstzīmēm garam",
      maxlength: "Uzvārds nedrīkst būt garāks par 25 rakstzīmēm",
    },
    password: {
      passwordRegex:
        'Parolei jāsastāv no 8 līdz 12 rakstzīmēm, ieskaitot burtus (A-Z, a-z) un ciparus (0-9). Bez īpašajiem simboliem (^@()#*+/"?!=.{}~`&) un atstarpēm',
      required: "Paroles lauks ir obligāts",
      minlength: "Parolei jābūt vismaz 8 rakstzīmēm garai",
      maxlength: "Parole nedrīkst būt garāka par 12 rakstzīmēm",
    },
    email: {
      required: "E-pasta lauks ir obligāts",
      email: "E-pastam jābūt derīgai adresei",
    },
    phone: {
      phoneRegex: "Tālrunim jābūt no 7 līdz 11 rakstzīmēm garam, bez īpašajām rakstzīmēm",
      required: "Tālrunis ir obligāts lauks",
    },
    checkbox: {
      text1: "Reģistrējoties es piekrītu un pieņemu mājaslapas lietošanas",
      link1: "noteikumus",
      text2: "un",
      link2: "Privātuma Politiku",
      text3: "",
    },
  },
  lt: {
    first_name: {
      usernameRegex: "Vardas turi būti ilgesnis nei 2 ženklai, be specialiųjų ženklų ir tarpų.",
      required: "Vardo laukas yra privalomas",
      minlength: "Vardas turi būti ne trumpesnis kaip 2 ženklai",
      maxlength: "Vardą gali sudaryti ne daugiau kaip 25 ženklai",
    },
    last_name: {
      usernameRegex: "Pavardė turi būti ilgesnė nei 2 ženklai, be specialiųjų ženklų ir tarpų.",
      required: "Pavardės laukas yra privalomas",
      minlength: "Pavardė turi būti ne trumpesnė kaip 2 ženklai",
      maxlength: "Pavardė gali būti ne daugiau kaip 25 ženklai",
    },
    password: {
      passwordRegex:
        'Slaptažodis turi būti 8-12 simbolių ilgio, įskaitant raides (A-Z, a-z) ir skaičius (0-9). Be jokių specialiųjų simbolių (^@()#*+/"?!=.{}~`&) ir tarpų.',
      required: "Slaptažodžio laukas yra privalomas",
      minlength: "Slaptažodį turi sudaryti ne mažiau kaip 8 simboliai",
      maxlength: "Slaptažodis negali būti ilgesnis nei 12 simbolių",
    },
    email: {
      required: "El. pašto laukas yra privalomas",
      email: "El. paštas turi būti galiojantis",
    },
    phone: {
      phoneRegex: "Telefonas turi būti nuo 7 iki 11 simbolių be specialiųjų ženklų.",
      required: "Telefonas yra privalomas",
    },
    checkbox: {
      text1: "Registruodamasis sutinku ir priimu svetainės",
      link1: "naudojimo",
      text2: "sąlygas ir",
      link2: "privatumo politiką",
      text3: "",
    },
  },
  ee: {
    first_name: {
      usernameRegex: "Nimi peab olema pikem kui 2 tähemärki, ilma erimärkide ja tühikuteta.",
      required: "Eesnime väli on kohustuslik",
      minlength: "Eesnimi peab olema vähemalt 2 tähemärki",
      maxlength: "Eesnimi võib olla maksimaalselt 25 tähemärki",
    },
    last_name: {
      usernameRegex: "Perekonnanimi peab olema pikem kui 2 tähemärki, ilma erimärkide ja tühikuteta",
      required: "Perekonnanime väli on kohustuslik",
      minlength: "Perekonnanimi peab olema vähemalt 2 tähemärki",
      maxlength: "Perekonnanimi võib olla maksimaalselt 25 tähemärki",
    },
    password: {
      passwordRegex:
        'Parool peab olema 8-12 tähemärki pikk, sealhulgas tähed (A-Z, a-z) ja numbrid (0-9). Ilma erisümboliteta (^@()#*+/"?!=.{}~`&) ja tühikuteta.',
      required: "Parooli väli on kohustuslik",
      minlength: "Parool peab olema vähemalt 8 tähemärki",
      maxlength: "Parool ei tohi olla pikem kui 12 tähemärki",
    },
    email: {
      required: "E-posti väli on kohustuslik",
      email: "E-post peab olema kehtiv aadress",
    },
    phone: {
      phoneRegex: "Telefon peab olema 7-11 tähemärki, ilma erimärkideta",
      required: "Telefoni number on kohustuslik",
    },
    checkbox: {
      text1: "Registreerimisega nõustun ja aktsepteerin veebisaidi",
      link1: "kasutustingimusi",
      text2: "ja",
      link2: "Privaatsuspoliitikat",
      text3: "",
    },
  },
  jp: {
    first_name: {
      usernameRegex: "名前は2文字以上で、特殊文字やスペースは使用しないでください。",
      required: "姓は必須項目です。",
      minlength: "姓は2文字以上でなければなりません",
      maxlength: "姓は最大25文字まで",
    },
    last_name: {
      usernameRegex: "姓は2文字以上で、特殊文字やスペースは使用できません。",
      required: "姓は必須項目です。",
      minlength: "姓は2文字以上でなければなりません",
      maxlength: "姓は最大25文字です。",
    },
    password: {
      passwordRegex:
        'パスワードは、英字（A-Z、a-z）および数字（0-9）を含む8～12文字で入力してください。特殊記号(^@()#*+/"?!=.{}~`&)およびスペースを含まないこと。',
      required: "パスワード・フィールドは必須です。",
      minlength: "パスワードは8文字以上でなければなりません。",
      maxlength: "パスワードは12文字以上であってはならない",
    },
    email: {
      required: "e-mailフィールドは必須です。",
      email: "E-mailは有効なアドレスでなければなりません",
    },
    phone: {
      phoneRegex: "電話番号は7文字以上11文字以内で、特殊文字は使用しないでください。",
      required: "電話番号は必須項目です。",
    },
    checkbox: {
      text1: "登録することにより、私はウェブサイトの",
      link1: "利用規約",
      text2: "と",
      link2: "プライバシーポリ",
      text3: "シーに同意し、承諾します。",
    },
  },
  kr: {
    first_name: {
      usernameRegex: "이름은 특수 문자나 공백 없이 2자 이상이어야 합니다.",
      required: "이름 필드는 필수입니다.",
      minlength: "이름은 2자 이상이어야 합니다.",
      maxlength: "이름은 최대 25자까지 입력 가능",
    },
    last_name: {
      usernameRegex: "성은 특수 문자나 공백 없이 2자 이상이어야 하며, 2자 이상이어야 합니다.",
      required: "성 필드는 필수 입력 사항입니다.",
      minlength: "성은 2자 이상이어야 합니다.",
      maxlength: "성은 최대 25자까지 입력할 수 있습니다.",
    },
    password: {
      passwordRegex:
        '비밀번호는 문자(A-Z, a-z)와 숫자(0-9)를 포함하여 8~12자 사이여야 합니다. 특수 기호(^@()#*+/"?!=.{}~`&)와 공백을 포함하지 않아야 합니다.',
      required: "비밀번호 필드는 필수입니다.",
      minlength: "비밀번호는 8자 이상이어야 합니다.",
      maxlength: "비밀번호는 12자를 초과할 수 없습니다.",
    },
    email: {
      required: "이메일 필드는 필수입니다.",
      email: "이메일은 유효한 주소여야 합니다.",
    },
    phone: {
      phoneRegex: "전화번호는 특수 문자 없이 7~11자 사이여야 합니다.",
      required: "전화는 필수 입력 사항입니다.",
    },
    checkbox: {
      text1: "등록함으로써 본인은 웹사이트",
      link1: "이용 약관",
      text2: "및",
      link2: "개인정보 보호정책",
      text3: "에 동의하고 수락합니다.",
    },
  },
  sr: {
    first_name: {
      usernameRegex: "Ime mora da sadrži najmanje 2 karaktera, bez posebnih znakova i razmaka",
      required: "Polje Ime je obavezno",
      minlength: "Ime mora da sadrži najmanje 2 karaktera",
      maxlength: "Ime može da sadrži najviše 25 karaktera",
    },
    last_name: {
      usernameRegex: "Prezime mora da sadrži najmanje 2 karaktera, bez posebnih znakova i razmaka",
      required: "Polje Prezime je obavezno",
      minlength: "Prezime mora da sadrži najmanje 2 karaktera",
      maxlength: "Prezime može da sadrži najviše 25 karaktera",
    },
    password: {
      passwordRegex:
        'Lozinka može da ima 8-12 karaktera, uključujući slova (A-Z, a-z) i brojeve (0-9). Bez posebnih simbola (^@()#*+/"?!=.{}~`&) i razmaka',
      required: "Polje Lozinka je obavezno",
      minlength: "Lozinka mora da sadrži najmanje 8 karaktera",
      maxlength: "Lozinka može da sadrži najviše 12 karaktera",
    },
    email: {
      required: "Polje e-mail je obavezno",
      email: "E-mail adresa mora biti važeća",
    },
    phone: {
      phoneRegex: "Telefon mora da sadrži 7 do 11 karaktera, bez posebnih karaktera",
      required: "Telefon je obavezan",
    },
    checkbox: {
      text1: "Registracijom prihvatam",
      link1: "uslove",
      text2: "korišćenja veb stranice i",
      link2: "Politiku privatnosti",
      text3: "",
    },
  },
  hi: {
    first_name: {
      usernameRegex: "नाम विशेष वर्ण या रिक्त स्थान के बिना, 2 अक्षर से अधिक लंबा होना चाहिए",
      required: "प्रथम नाम फ़ील्ड आवश्यक है",
      minlength: "पहला नाम कम से कम 2 अक्षर का होना चाहिए",
      maxlength: "प्रथम नाम अधिकतम 25 अक्षर का हो सकता है",
    },
    last_name: {
      usernameRegex: "अंतिम नाम विशेष वर्ण या रिक्त स्थान के बिना, 2 अक्षर से अधिक लंबा होना चाहिए",
      required: "अंतिम नाम फ़ील्ड आवश्यक है",
      minlength: "अंतिम नाम कम से कम 2 अक्षर का होना चाहिए",
      maxlength: "उपनाम अधिकतम 25 अक्षर का हो सकता है",
    },
    password: {
      passwordRegex:
        'पासवर्ड की लंबाई 8-12 अक्षरों के बीच होनी चाहिए, जिसमें अक्षर (AZ, az) और अंक (0-9) शामिल हैं। बिना किसी विशेष चिह्न (^@()#*+/"?!=.{}~`&) और रिक्त स्थान के।',
      required: "पासवर्ड फ़ील्ड आवश्यक है",
      minlength: "पासवर्ड कम से कम 8 अक्षरों का होना चाहिए",
      maxlength: "पासवर्ड 12 अक्षरों से बड़ा नहीं होना चाहिए",
    },
    email: {
      required: "ईमेल क्षेत्र की आवश्यकता है",
      email: "ई-मेल पता वैध होना चाहिए",
    },
    phone: {
      phoneRegex: "फ़ोन में 7 से 11 अक्षर होने चाहिए, विशेष अक्षर नहीं होने चाहिए",
      required: "फ़ोन ज़रूरी है",
    },
    checkbox: {
      text1: "रजिस्ट्रेशन द्वारा मैं वेबसाइट के उपयोग की",
      link1: "शर्तों",
      text2: "और",
      link2: "गोपनीयता नीति",
      text3: "से सहमत हूं और स्वीकार करता हूं",
    },
  },
}

const useThanksPage = true 
const useCheckbox = true 
const usePreloader = true
const checkboxAfterSubmit = true
const usePhoneccField = true // add hiden input field
const useSessionIdField = true // add hiden input field
const useAffiliateIdField = true // add hiden input field

var lang = navigator.languages
var ref =  document.referrer
var forms = document.getElementsByTagName('form');
    for (var i = 0; i < forms.length; i++) {
        var input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'lang';
        input.value = lang;
        forms[i].appendChild(input);
    }
    for (var i = 0; i < forms.length; i++) {
        var input = document.createElement('input');
        input.type = 'hidden';
        input.name = 'ref';
        input.value = ref;
        forms[i].appendChild(input);
    }

    (function () {

    function getSearchParameters() {
        var prmstr = window.location.search.substr(1);
        return prmstr != null && prmstr != "" ? transformToAssocArray(prmstr) : {};
    }

    function transformToAssocArray(prmstr) {
        var params = {};
        var prmarr = prmstr.split("&");
        for (var i = 0; i < prmarr.length; i++) {
            var tmparr = prmarr[i].split("=");
            params[tmparr[0]] = decodeURIComponent(tmparr[1]);
        }
        return params;
    }

    var params = getSearchParameters();

    Object.keys(params).map(function (key, index) {
        for (i = 0; i < document.forms.length; i++) {
            var hiddenField = document.createElement("input");
            hiddenField.setAttribute("type", "hidden");
            hiddenField.setAttribute("name", key);
            hiddenField.setAttribute("value", params[key]);
            document.forms[i].appendChild(hiddenField);
        }
    });


})();
